package com.amity.socialcloud.sdk.api.social.community

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.social.category.query.AmityCommunityCategoryQuery
import com.amity.socialcloud.sdk.api.social.community.create.AmityCommunityCreator
import com.amity.socialcloud.sdk.api.social.community.query.AmityCommunityQuery
import com.amity.socialcloud.sdk.api.social.community.search.AmityCommunitySearch
import com.amity.socialcloud.sdk.api.social.community.update.AmityCommunityUpdate
import com.amity.socialcloud.sdk.api.social.member.AmityCommunityModeration
import com.amity.socialcloud.sdk.api.social.member.AmityCommunityParticipation
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.category.AmityCommunityCategory
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityMembershipStatusFilter
import com.amity.socialcloud.sdk.model.social.community.AmityJoinRequest
import com.amity.socialcloud.sdk.social.domain.category.get.CategoryLiveObjectUseCase
import com.amity.socialcloud.sdk.social.domain.community.CommunityLiveObjectUseCase
import com.amity.socialcloud.sdk.social.domain.community.DeleteCommunityUseCase
import com.amity.socialcloud.sdk.social.domain.community.GetLatestJoinRequestUseCase
import com.amity.socialcloud.sdk.social.domain.community.JoinCommunityUseCase
import com.amity.socialcloud.sdk.social.domain.community.LeaveCommunityUseCase
import com.amity.socialcloud.sdk.social.domain.community.RecommendedCommunityQueryUseCase
import com.amity.socialcloud.sdk.social.domain.community.SemanticSearchCommunitiesUseCase
import com.amity.socialcloud.sdk.social.domain.community.TrendingCommunityQueryUseCase
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

class AmityCommunityRepository internal constructor() {

    /* begin_public_function
	  id: community.create
	  api_style: async
	*/
    /**
     * Initiates a function chain to create a community.
     * @param [displayName] a community's displayName
     * @return new instance of AmityCommunityCreator.Builder.
     */
    fun createCommunity(
        displayName: String,
        isDiscoverable: Boolean? = null,
        requiresJoinApproval: Boolean? = null,
    ): AmityCommunityCreator.Builder {
        return AmityCommunityCreator.Builder(displayName, isDiscoverable, requiresJoinApproval)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.query
	  api_style: live_collection
	*/
    /**
     * Initiates a function chain to query for communities.
     * @return new instance of AmityCommunityQuery.Builder.
     */
    fun getCommunities(includeDiscoverablePrivateCommunity: Boolean = false): AmityCommunityQuery.Builder {
        return AmityCommunityQuery.Builder(includeDiscoverablePrivateCommunity)
    }
    /* end_public_function */

    /* begin_public_function
    id: community.getLatestJoinRequest
    api_style: live_collection
  */
    /**
     * Initiates a function chain to query for communities.
     * @return new instance of AmityCommunityQuery.Builder.
     */
    fun getLatestJoinRequest(communityIds:List<String>): Single<AmityJoinRequest> {
        return GetLatestJoinRequestUseCase().execute(communityIds = communityIds)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.membership.search
	  api_style: live_collection
	*/
    /**
     * Initiates a function chain to search for communities.
     * @param [keyword] value to search by community's displayName
     * @return new instance of AmityCommunitySearch.Builder.
     */
    fun searchCommunities(keyword: String, includeDiscoverablePrivateCommunity: Boolean = false): AmityCommunitySearch.Builder {
        return AmityCommunitySearch.Builder(keyword, includeDiscoverablePrivateCommunity)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.semanticSearch
	  api_style: live_collection
	*/
    fun semanticSearchCommunities(
        query: String,
        filter: AmityCommunityMembershipStatusFilter = AmityCommunityMembershipStatusFilter.ALL,
        tags: AmityTags = AmityTags(),
        categoryIds: List<String> = emptyList(),
        includeDiscoverablePrivateCommunity: Boolean = false
    ): Flowable<PagingData<AmityCommunity>> {
        return SemanticSearchCommunitiesUseCase().execute(
            query = query,
            filter = filter,
            tags = tags,
            categoryIds = categoryIds,
            includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.query.recommended_communities
	*/
    /**
     * Queries for recommended communities.
     * @return LiveCollection of posts as Flowable<List<AmityCommunity>>.
     */
    fun getRecommendedCommunities(includeDiscoverablePrivateCommunity: Boolean = false): Flowable<List<AmityCommunity>> {
        return RecommendedCommunityQueryUseCase().execute(includeDiscoverablePrivateCommunity)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.query.trending_communities
	*/
    /**
     * Queries for trending communities.
     * @return LiveCollection of posts as Flowable<List<AmityCommunity>>.
     */
    fun getTrendingCommunities(includeDiscoverablePrivateCommunity: Boolean = false): Flowable<List<AmityCommunity>> {
        return TrendingCommunityQueryUseCase().execute(includeDiscoverablePrivateCommunity)
    }
    /* end_public_function */

    /**
     * Instantiates AmityCommunityParticipation.
     * @param [communityId] a communityId
     * @return new instance of AmityCommunityParticipation.
     */
    fun membership(communityId: String): AmityCommunityParticipation {
        return AmityCommunityParticipation(communityId)
    }

    /* begin_public_function
	  id: community.get
	*/
    /**
     * Gets a LiveObject of community
     * @param [communityId] a communityId
     * @return LiveObject of community as Flowable<AmityCommunity>
     */
    fun getCommunity(communityId: String): Flowable<AmityCommunity> {
        return CommunityLiveObjectUseCase().execute(communityId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.category.get
	*/
    /**
     * Gets a LiveObject of category
     * @param [categoryId] a categoryId
     * @return LiveObject of category as Flowable<AmityCommunityCategory>
     */
    fun getCategory(categoryId: String): Flowable<AmityCommunityCategory> {
        return CategoryLiveObjectUseCase().execute(categoryId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.category.query
	  api_style: live_collection
	*/
    /**
     * Initiates a function chain to query for categories.
     * @return new instance of AmityCommunityCategoryQuery.Builder.
     */
    fun getCategories(): AmityCommunityCategoryQuery.Builder {
        return AmityCommunityCategoryQuery.Builder()
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.delete
	*/
    /**
     * deletes a community
     * @param [communityId] a communityId
     * @return Async completion as Completable
     */
    fun deleteCommunity(communityId: String): Completable {
        return DeleteCommunityUseCase().execute(communityId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.update
	  api_style: async
	*/
    /**
     * Initiates a function chain to edit a community.
     * @return new instance of AmityCommunityUpdate.Builder.
     */
    fun editCommunity(
        communityId: String,
        isDiscoverable: Boolean? = null,
        requiresJoinApproval: Boolean? = null,
    ): AmityCommunityUpdate.Builder {
        return AmityCommunityUpdate.Builder(communityId, isDiscoverable, requiresJoinApproval)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.join
	*/
    /**
     * joins a community
     * @param [communityId] a communityId
     * @return Async completion as Completable
     */
    @Deprecated("Use AmityCommunity.join() instead")
    fun joinCommunity(communityId: String): Completable {
        return JoinCommunityUseCase().execute(communityId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: community.leave
	*/
    /**
     * leaves a community
     * @param [communityId] a communityId
     * @return Async completion as Completable
     */
    fun leaveCommunity(communityId: String): Completable {
        return LeaveCommunityUseCase().execute(communityId)
    }
    /* end_public_function */

    /**
     * Instantiates AmityCommunityModeration.
     * @param [communityId] a communityId
     * @return new instance of AmityCommunityModeration.
     */
    fun moderation(communityId: String): AmityCommunityModeration {
        return AmityCommunityModeration(communityId)
    }

}