package com.amity.socialcloud.sdk.social.data.story

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.social.story.AmityStorySortOption
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.entity.social.story.StoryEntity
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryImageDisplayMode
import com.amity.socialcloud.sdk.model.social.story.AmityStoryItem
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.SinglePagePagerCreator
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import org.joda.time.DateTime
import java.util.UUID


@OptIn(ExperimentalPagingApi::class)
internal class StoryRepository : AmityObjectRepository<StoryEntity, AmityStory>() {

    override fun fetchAndSave(objectId: String): Completable {
        return StoryRemoteDataStore().getStory(objectId)
            .flatMapCompletable {
                StoryQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): StoryEntity? {
        val story = StoryLocalDataStore().getStoryNow(objectId)
        // We intentionally don't support unsynced object as a live object
        // such as optimistic object (pre-created)
        if (story != null
            && AmityStory.State.enumOf(story.syncState) != AmityStory.State.SYNCED
        ) {
            throw   AmityException.create(
                message = "Observing unsynced object is not supported by Live Object.",
                cause = null,
                error = AmityError.UNSUPPORTED
            )
        }
        return story
    }

    override fun mapper(): ModelMapper<StoryEntity, AmityStory> {
        return StoryModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<StoryEntity> {
        return StoryLocalDataStore().observeStory(objectId)
    }

    fun generateUniqueId(): String {
        return "LOCAL_" + UUID.randomUUID().toString()
    }

    fun createLocalImageStory(
        referenceId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject?,
        imageDisplayMode: AmityStoryImageDisplayMode
    ): Completable {
        return StoryLocalDataStore().createImageStory(
            referenceId = referenceId,
            targetType = targetType,
            targetId = targetId,
            storyItems = storyItems,
            metadata = metadata,
            imageDisplayMode = imageDisplayMode
        )
    }

    fun createImageStory(
        referenceId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        fileId: String,
        storyItems: List<AmityStoryItem>,
        metadata: JsonObject?,
        imageDisplayMode: AmityStoryImageDisplayMode
    ): Single<String> {
        return StoryRemoteDataStore().createImageStory(
            referenceId = referenceId,
            targetType = targetType,
            targetId = targetId,
            fileId = fileId,
            storyItems = storyItems,
            metadata = metadata,
            imageDisplayMode = imageDisplayMode
        )
            .flatMap {
                StoryQueryPersister().persist(it)
                    .andThen(Single.just(it.stories?.first()?.storyId ?: ""))
            }

    }

    fun updateStorySyncState(
        referenceId: String,
        syncState: AmityStory.State
    ): Completable {
        return StoryLocalDataStore().updateStorySyncState(
            referenceId = referenceId,
            syncState = syncState
        )
    }

    fun createLocalVideoStory(
        referenceId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        storyItems: List<AmityStoryItem>,
        metadata: JsonObject?
    ): Completable {
        return StoryLocalDataStore().createVideoStory(
            referenceId = referenceId,
            targetType = targetType,
            targetId = targetId,
            storyItems = storyItems,
            metadata = metadata
        )
    }

    fun createVideoStory(
        referenceId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        fileId: String,
        storyItems: List<AmityStoryItem>,
        metadata: JsonObject?
    ): Single<String> {
        return StoryRemoteDataStore().createVideoStory(
            referenceId = referenceId,
            targetType = targetType,
            targetId = targetId,
            fileId = fileId,
            storyItems = storyItems,
            metadata = metadata
        )
            .flatMap {
                StoryQueryPersister().persist(it)
                    .andThen(Single.just(it.stories?.first()?.storyId ?: ""))
            }

    }

    fun getActiveStories(
        targetType: AmityStory.TargetType,
        targetId: String,
        sortOption: AmityStorySortOption
    ): Flowable<PagingData<AmityStory>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = 100,
                enablePlaceholders = true,
                initialLoadSize = 100,
                prefetchDistance = 0
            ),
            dynamicQueryStreamMediator = StoryMediator(
                targetType = targetType,
                targetId = targetId,
                sortOption = sortOption
            ),
            pagingSourceFactory = {
                StoryLocalDataStore().getStoryPagingSource(targetType, targetId, sortOption)
            },
            modelMapper = StoryModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun deleteStory(storyId: String, permanent: Boolean): Completable {
        val deleteCache = StoryLocalDataStore().deleteStory(storyId)
        return if (storyId.startsWith("LOCAL_")) {
            deleteCache
        } else {
            StoryRemoteDataStore().deleteStory(storyId, permanent)
                .ignoreElement()
                .andThen(deleteCache)
        }
    }

    internal fun getLatestStories(
        targetType: AmityStory.TargetType,
        targetId: String,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<AmityStory> {
        return StoryLocalDataStore().getLatestStories(
            targetType,
            targetId,
            dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce
        )
            .map {
                StoryModelMapper().map(it)
            }
    }

    fun getStoryNow(storyId: String): AmityStory? {
        return StoryLocalDataStore().getStoryNow(storyId)?.let {
            StoryModelMapper().map(it)
        }
    }

    fun getSyncingStoriesCount(
        targetType: AmityStory.TargetType,
        targetId: String
    ): Int {
        return StoryLocalDataStore().getStoryCount(targetType, targetId, listOf(AmityStory.State.SYNCING))
    }

    fun getFailedStoriesCount(
        targetType: AmityStory.TargetType,
        targetId: String
    ): Int {
        return StoryLocalDataStore().getStoryCount(targetType, targetId, listOf(AmityStory.State.FAILED))
    }


    fun getHighestStoryExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String
    ): DateTime? {
        return StoryLocalDataStore().getHighestStoryExpiresAt(targetType, targetId, listOf(AmityStory.State.FAILED, AmityStory.State.SYNCING))
    }

    fun getHighestSyncedStoryExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String
    ): DateTime? {
        return StoryLocalDataStore().getHighestStoryExpiresAt(targetType, targetId, listOf(AmityStory.State.SYNCED))
    }

    fun triggerStoryReload(storyId: String): Completable {
        return StoryLocalDataStore().triggerStoryReload(storyId)
    }

    fun findCache(targetType: AmityStory.TargetType, targetId: String) : Single<List<String>> {
        return StoryLocalDataStore().findCache(targetType, targetId)
    }

    fun getStoriesByTargets(
        targets: List<Pair<AmityStory.TargetType, String>>,
        orderBy: AmityStorySortOption
    ): Flowable<List<AmityStory>> {
        val pagerCreator = SinglePagePagerCreator(
            mediator = StorySinglePageMediator(
                targets = targets,
                orderBy = orderBy
            ),
            domainDatasource = StoryLocalDataStore().getStoriesByTargets(targets),
            modelMapper = StoryModelMapper()
        )

        return pagerCreator.create()
    }

}
