package com.amity.socialcloud.sdk.api.social.story

import android.net.Uri
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.social.storytarget.AmityGlobalStoryTargetsQueryOption
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryImageDisplayMode
import com.amity.socialcloud.sdk.model.social.story.AmityStoryItem
import com.amity.socialcloud.sdk.model.social.story.AmityStoryTarget
import com.amity.socialcloud.sdk.social.domain.story.StoryLiveObjectUseCase
import com.amity.socialcloud.sdk.social.domain.story.create.CreateImageStoryUseCase
import com.amity.socialcloud.sdk.social.domain.story.create.CreateVideoStoryUseCase
import com.amity.socialcloud.sdk.social.domain.story.delete.DeleteStoryUseCase
import com.amity.socialcloud.sdk.social.domain.story.query.GetActiveStoriesUseCase
import com.amity.socialcloud.sdk.social.domain.story.query.GetStoriesByTargetsUseCase
import com.amity.socialcloud.sdk.social.domain.storytarget.get.GetGlobalStoryTargetsUseCase
import com.amity.socialcloud.sdk.social.domain.storytarget.get.GetStoryTargetsUseCase
import com.amity.socialcloud.sdk.social.domain.storytarget.get.StoryTargetLiveObjectUseCase
import com.google.gson.JsonObject

import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable

class AmityStoryRepository internal constructor() {

    fun createImageStory(
        targetType: AmityStory.TargetType,
        targetId: String,
        imageUri: Uri,
        storyItems: List<AmityStoryItem> = emptyList(),
        imageDisplayMode: AmityStoryImageDisplayMode = AmityStoryImageDisplayMode.FIT,
        metadata: JsonObject? = null
    ): Completable {
        return CreateImageStoryUseCase().execute(
            targetType = targetType,
            targetId = targetId,
            fileUri = imageUri,
            storyItems = storyItems,
            metadata = metadata,
            imageDisplayMode = imageDisplayMode,
        )
    }

    fun createVideoStory(
        targetType: AmityStory.TargetType,
        targetId: String,
        videoUri: Uri,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject? = null
    ): Completable {
        return CreateVideoStoryUseCase().execute(
            targetType = targetType,
            targetId = targetId,
            fileUri = videoUri,
            storyItems = storyItems,
            metadata = metadata
        )
    }

    fun getStoryTarget(
        targetType: AmityStory.TargetType,
        targetId: String
    ): Flowable<AmityStoryTarget> {
        return StoryTargetLiveObjectUseCase().execute(
            AmityStoryTarget.generateUniqueId(targetType.apiKey, targetId)
        )
    }

    fun getStoryTargets(
        targets: List<Pair<AmityStory.TargetType, String>>
    ): Flowable<List<AmityStoryTarget>> {
        return GetStoryTargetsUseCase().execute(targets)
    }

    fun getGlobalStoryTargets(queryOption: AmityGlobalStoryTargetsQueryOption): Flowable<PagingData<AmityStoryTarget>> {
        return GetGlobalStoryTargetsUseCase().execute(queryOption)
    }

    fun getActiveStories(
        targetType: AmityStory.TargetType,
        targetId: String,
        sortOption: AmityStorySortOption = AmityStorySortOption.FIRST_CREATED,
    ): Flowable<PagingData<AmityStory>> {
        return GetActiveStoriesUseCase().execute(
            targetType = targetType,
            targetId = targetId,
            sortOption = sortOption
        )
    }

    fun getStoriesByTargets(
        targets: List<Pair<AmityStory.TargetType, String>>,
        sortOption: AmityStorySortOption = AmityStorySortOption.FIRST_CREATED,
    ): Flowable<List<AmityStory>> {
        return GetStoriesByTargetsUseCase().execute(
            targets = targets,
            sortOption = sortOption
        )
    }

    fun getStory(storyId: String): Flowable<AmityStory> {
        return StoryLiveObjectUseCase().execute(storyId)
    }

    fun softDeleteStory(storyId: String): Completable {
        return DeleteStoryUseCase().execute(storyId, false)
    }

    fun hardDeleteStory(storyId: String): Completable {
        return DeleteStoryUseCase().execute(storyId, true)
    }

    fun analytics(story: AmityStory): AmityStoryAnalytics {
        return AmityStoryAnalytics(story)
    }

}