package com.amity.socialcloud.sdk.video.data.stream

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.model.video.stream.AmityStream
import com.amity.socialcloud.sdk.model.video.stream.AmityStreamModeration
import com.amity.socialcloud.sdk.video.data.moderation.StreamModerationLocalDataStore
import com.amity.socialcloud.sdk.video.data.moderation.StreamModerationModelMapper
import com.amity.socialcloud.sdk.video.model.AmityBroadcastResolution
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.api.socket.request.CreateStreamRequest
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

internal class StreamRepository : EkoObjectRepository() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    @OptIn(ExperimentalPagingApi::class)
    fun getStreamCollectionPagingData(statuses: Array<AmityStream.Status>): Flowable<PagingData<AmityStream>> {
        val statusList = getFormattedStatuses(statuses).toList()
        val pagerCreator = DynamicQueryStreamPagerCreator(
                pagingConfig = PagingConfig(
                        pageSize = getDefaultPageSize(),
                        enablePlaceholders = false
                ),
                dynamicQueryStreamMediator = StreamMediator(
                    statusList
                ),
                pagingSourceFactory = {
                    StreamLocalDataStore().getStreamsPagingSource(statusList)
                },
                modelMapper = StreamModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun observeStream(streamId: String): Flowable<AmityStream> {
        return StreamLocalDataStore().observeStream(streamId)
            .map {
                StreamModelMapper().map(it)
            }
    }

    fun fetchStream(streamId: String): Completable {
        return StreamRemoteDataStore().getVideoStreaming(streamId)
            .flatMapCompletable {
                StreamQueryPersister().persist(it)
            }
    }

    fun createStream(
        title: String,
        description: String,
        metadata: JsonObject? = null,
        resolution: AmityBroadcastResolution,
        thumbnailFileId: String? = null
    ): Single<String> {
        val request = CreateStreamRequest(
            title = title,
            description = description,
            resolution = resolution.readableName,
            thumbnailFileId = thumbnailFileId,
            metadata = metadata,
        )

        return StreamRemoteDataStore().createVideoStreaming(request)
            .flatMap {
                StreamQueryPersister().persist(it)
                    .andThen(Single.just(it.streams.first().streamId))
            }
    }

    fun createStreamBroadcaster(
        title: String,
        description: String,
        resolution: AmityBroadcastResolution,
        thumbnailFileId: String? = null
    ): Single<Pair<String, String?>> {
        return createStream(title, description, null, resolution, thumbnailFileId)
            .flatMap {
                observeStream(it).firstOrError()
            }
            .map {
                val streamId: String = it.getStreamId()
                val broadcastingUrl: String? = it.getBroadcasterData()?.getUrl()
                (streamId to broadcastingUrl)
            }
    }

    fun disposeVideoStreaming(streamId: String): Completable {
        return StreamRemoteDataStore().disposeVideoStreaming(streamId).ignoreElement()
    }

    private fun getFormattedStatuses(statuses: Array<AmityStream.Status>): Array<String> {
        val statusSet = if (statuses.isNotEmpty()) {
            statuses.toSet()
        } else {
            setOf(AmityStream.Status.IDLE)
        }
        return statusSet.map(AmityStream.Status::apiKey).toTypedArray()
    }

    fun getLatestSteam(statuses: List<String>): Flowable<AmityStream> {
        return StreamLocalDataStore().getLatestStream(statuses = statuses)
                .map(StreamModelMapper()::map)
    }

    fun getStreamModeration(moderationId: String) : AmityStreamModeration? {
        return StreamModerationLocalDataStore().getStreamModeration(moderationId)?.let {
            StreamModerationModelMapper().map(it)
        }
    }

}