package com.amity.socialcloud.sdk.social.data.story

import com.amity.socialcloud.sdk.api.social.story.AmityStorySortOption
import com.amity.socialcloud.sdk.dto.social.story.CreateImageStoryRequest
import com.amity.socialcloud.sdk.dto.social.story.CreateVideoStoryRequest
import com.amity.socialcloud.sdk.dto.social.story.StoryQueryDto
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryImageDisplayMode
import com.amity.socialcloud.sdk.model.social.story.AmityStoryItem
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.EkoDeletionDto
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single

internal class StoryRemoteDataStore {

    fun createImageStory(
        referenceId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        fileId: String,
        storyItems: List<AmityStoryItem>,
        metadata: JsonObject?,
        imageDisplayMode: AmityStoryImageDisplayMode
    ): Single<StoryQueryDto> {
        return Single.fromCallable {
            CreateImageStoryRequest(
                referenceId = referenceId,
                targetType = targetType.apiKey,
                targetId = targetId,
                fileId = fileId,
                storyItems = storyItems,
                imageDisplayMode = imageDisplayMode,
                metadata = metadata
            )
        }.flatMap { request ->
            AmityHttpClient.get(StoryApi::class).flatMap {
                it.createStory(request)
            }
        }
    }

    fun createVideoStory(
        referenceId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        fileId: String,
        storyItems: List<AmityStoryItem>,
        metadata: JsonObject?
    ): Single<StoryQueryDto> {
        return Single.fromCallable {
            CreateVideoStoryRequest(
                referenceId = referenceId,
                targetType = targetType.apiKey,
                targetId = targetId,
                fileId = fileId,
                storyItems = storyItems,
                metadata = metadata
            )
        }.flatMap { request ->
            AmityHttpClient.get(StoryApi::class).flatMap {
                it.createStory(request)
            }
        }
    }

    fun getStories(
        targetType: AmityStory.TargetType,
        targetId: String,
        orderBy: String,
    ): Single<StoryQueryDto> {
        return AmityHttpClient.get(StoryApi::class).flatMap {
            it.queryStories(
                targetType = targetType.apiKey,
                targetId = targetId,
                sortBy = "createdAt",
                orderBy = orderBy,
            )
        }
    }

    fun getStoriesByTargets(
        targets: List<Pair<AmityStory.TargetType, String>>,
        orderBy: AmityStorySortOption
    ): Single<StoryQueryDto> {

        return Single.fromCallable {
            mutableMapOf<String,String>().apply {
                targets.forEach {
                    put("targets[${targets.indexOf(it)}][targetType]", it.first.apiKey)
                    put("targets[${targets.indexOf(it)}][targetId]", it.second)
                }
            }
        }.flatMap { map ->
            AmityHttpClient.get(StoryApi::class).flatMap {
                it.queryStoriesByTargets(
                    targets = map,
                    sortBy = "createdAt",
                    orderBy = if(orderBy == AmityStorySortOption.FIRST_CREATED) "asc" else "desc",
                )
            }
        }
    }

    fun getStory(
       storyId: String,
    ): Single<StoryQueryDto> {
        return AmityHttpClient.get(StoryApi::class).flatMap {
            it.getStory(
                storyId = storyId
            )
        }
    }

    fun deleteStory(storyId: String, permanent: Boolean): Single<EkoDeletionDto> {
        return AmityHttpClient.get(StoryApi::class).flatMap {
            it.deleteStory(storyId, permanent)
        }
    }

}