package com.amity.socialcloud.sdk.api.social.comment

import com.amity.socialcloud.sdk.api.core.ExperimentalAmityApi
import com.amity.socialcloud.sdk.api.social.comment.create.AmityCommentCreationTarget
import com.amity.socialcloud.sdk.api.social.comment.latest.AmityLatestCommentQueryType
import com.amity.socialcloud.sdk.api.social.comment.query.AmityCommentQueryType
import com.amity.socialcloud.sdk.api.social.comment.update.AmityAttachmentCommentUpdate
import com.amity.socialcloud.sdk.model.core.flag.AmityContentFlagReason
import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.amity.socialcloud.sdk.social.domain.comment.getbyids.CommentGetByIdsUseCase
import com.ekoapp.ekosdk.internal.api.socket.request.FlagContentRequest
import com.ekoapp.ekosdk.internal.usecase.comment.DeleteCommentUseCase
import com.ekoapp.ekosdk.internal.usecase.comment.FlagCommentUseCase
import com.ekoapp.ekosdk.internal.usecase.comment.ObserveSingleCommentUseCase
import com.ekoapp.ekosdk.internal.usecase.comment.UnFlagCommentUseCase
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable

class AmityCommentRepository internal constructor() {
    
    /* begin_public_function
      id: comment.get
    */
    /**
     * Gets a LiveObject of comment
     * @param [commentId] a commentId
     * @return LiveObject of comment as Flowable<AmityComment>
     */
    @ExperimentalAmityApi
    fun getComment(commentId: String): Flowable<AmityComment> {
        return ObserveSingleCommentUseCase().execute(commentId)
    }
    /* end_public_function */

    /* begin_public_function
     id: comment.get_by_ids
     api_style: live_collection
    */
    /**
     * Gets a LiveCollection of comments
     * @param [commentIds] an array of comment
     * @return LiveCollection of comment as Flowable<List<AmityComment>>
     */
    fun getCommentByIds(commentIds: Set<String>): Flowable<List<AmityComment>> {
        return CommentGetByIdsUseCase().execute(commentIds)
    }
    /* end_public_function */
    
    /* begin_public_function
      id: comment.query
      api_style: live_collection
    */
    /**
     * Initiates a function chain to query for comments.
     * @return new instance of AmityCommentQueryType.
     */
    fun getComments(): AmityCommentQueryType {
        return AmityCommentQueryType()
    }
    /* end_public_function */
    
    /* begin_public_function
      id: comment.create
      api_style: async
    */
    /**
     * Initiates a function chain to create a comment.
     * @return new instance of AmityCommentCreationTarget.
     */
    fun createComment(): AmityCommentCreationTarget {
        return AmityCommentCreationTarget()
    }
    /* end_public_function */
    
    /* begin_public_function
      id: comment.update_comment
      api_style: async
    */
    /**
     * Initiates a function chain to edit a comment.
     * @param [commentId] a commentId
     * @return new instance of AmityAttachmentCommentUpdate.Builder.
     */
    fun editComment(commentId: String): AmityAttachmentCommentUpdate.Builder{
        return AmityAttachmentCommentUpdate.Builder(commentId)
    }
    /* end_public_function */
    
    /* begin_public_function
      id: comment.soft_delete
    */
    /**
     * soft deletes a comment
     * @param [commentId] a commentId
     * @return Async completion as Completable
     */
    fun softDeleteComment(commentId: String): Completable {
        return DeleteCommentUseCase().execute(commentId, false)
    }
    /* end_public_function */
    
    /* begin_public_function
      id: comment.hard_delete
    */
    /**
     * hard deletes a comment
     * @param [commentId] a commentId
     * @return Async completion as Completable
     */
    fun hardDeleteComment(commentId: String): Completable {
        return DeleteCommentUseCase().execute(commentId, true)
    }
    /* end_public_function */

    
    /* begin_public_function
      id: comment.get_latest_comment
      api_style: async
    */
    /**
     * Initiates a function chain to query for latest comments.
     * @return new instance of AmityLatestCommentQueryType.
     */
    fun getLatestComment(): AmityLatestCommentQueryType {
        return AmityLatestCommentQueryType()
    }
    /* end_public_function */
    
    /* begin_public_function
      id: comment.flag
    */
    /**
     * flags a comment
     * @param [commentId] a commentId
     * @return Async completion as Completable
     */
    fun flagComment(commentId: String, reason: AmityContentFlagReason): Completable {
        val request = when(reason){
            is AmityContentFlagReason.Others -> {
                FlagContentRequest(
                    reason = reason.reason,
                    details = reason.details
                )
            }
            else -> {
                FlagContentRequest(
                    reason = reason.reason,
                )
            }
        }
        return FlagCommentUseCase().execute(commentId, request)
    }
    /* end_public_function */
    
    /* begin_public_function
      id: comment.unflag
    */
    /**
     * unflags a comment
     * @param [commentId] a commentId
     * @return Async completion as Completable
     */
    fun unflagComment(commentId: String): Completable {
        return UnFlagCommentUseCase().execute(commentId)
    }
    /* end_public_function */

    

}