package com.amity.socialcloud.sdk.social.data.community.paging

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.api.social.community.query.AmityCommunitySortOption
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.CommunityRemoteDataStore
import com.amity.socialcloud.sdk.social.data.community.CommunityRepository
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

internal class CommunityQueryMediator(
    private val keyword: String,
    private val categoryId: String,
    private val filter: AmityCommunityFilter,
    private val sortOption: AmityCommunitySortOption,
    private val isDeleted: Boolean?,
) : DynamicQueryStreamMediator<CommunityEntity, CommunityListQueryDto, AmityCommunity>(
    nonce = AmityNonce.COMMUNITY_LIST,
    dynamicQueryStreamKeyCreator = CommunityQueryKeyCreator(
        keyword = keyword,
        categoryId = categoryId,
        filter = filter,
        sortOption = sortOption,
        isDeleted = isDeleted
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityCommunity> {
        return if(keyword.isNotBlank()) {
            Flowable.empty()
        } else {
           CommunityRepository().getLatestCommunity(
               categoryId,
               filter,
               isDeleted,
               dynamicQueryStreamKeyCreator,
               nonce
           ).toRx2()
        }
    }

    override fun getFirstPageRequest(pageSize: Int): Single<CommunityListQueryDto> {
        return getRequest(limit = pageSize)

    }

    override fun getFetchByTokenRequest(token: String): Single<CommunityListQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: CommunityListQueryDto): Completable {
        return CommunityQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: CommunityListQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.communities?.map { it.communityId ?: "" } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<CommunityListQueryDto> {
        return CommunityRemoteDataStore().queryCommunities(
            keyword = keyword.ifEmpty { null },
            categoryId = categoryId.ifEmpty { null },
            filter = filter.value,
            sortBy = sortOption.apiKey,
            isDeleted = isDeleted,
            options = QueryOptionsRequestParams(
                limit = limit,
                token = token
            )
        ).toRx2()
    }

}