package com.ekoapp.ekosdk.internal.data.dao

import androidx.sqlite.db.SimpleSQLiteQuery
import com.ekoapp.ekosdk.internal.data.model.EkoQueryToken
import io.reactivex.Completable
import io.reactivex.Maybe
import io.reactivex.schedulers.Schedulers

interface AmityPagingTokenDao<QUERY_TOKEN : EkoQueryToken> {

    fun getFirstQueryToken(primaryKeys: Map<String, Any>): Maybe<QUERY_TOKEN> {
        return queryToken(
            SimpleSQLiteQuery(
                String.format(
                    "select * from %s where %s order by pageNumber asc limit 1",
                    tableName(), condition(primaryKeys)
                )
            )
        ).filter { it.previous != null }
    }

    fun getLastQueryToken(primaryKeys: Map<String, Any>): Maybe<QUERY_TOKEN> {
        return queryToken(
            SimpleSQLiteQuery(
                String.format(
                    "select * from %s where %s order by pageNumber desc limit 1",
                    tableName(), condition(primaryKeys)
                )
            )
        ).filter { it.next != null }
    }

    fun deleteObsoleteQueryToken(primaryKeys: Map<String, Any>, lastValidPage: Int) {
        queryToken(
                SimpleSQLiteQuery(
                        String.format(
                                "delete from %s where %s and pageNumber > %d",
                                tableName(), condition(primaryKeys), lastValidPage
                        )
                )
        ).subscribeOn(Schedulers.io())
                .subscribe()
    }

    @Deprecated("for backward compat only!")
    fun getQueryTokenByPageNumber(primaryKeys: Map<String, Any>, pageNumber: Int): Maybe<QUERY_TOKEN> {
        return queryToken(
            SimpleSQLiteQuery(
                String.format(
                    "select * from %s where %s and pageNumber = %s limit 1",
                    tableName(),
                    condition(primaryKeys),
                    pageNumber
                )
            )
        )
    }

    fun queryToken(query: SimpleSQLiteQuery): Maybe<QUERY_TOKEN>

    fun insertToken(token: QUERY_TOKEN): Completable

    fun tableName(): String

    private fun condition(primaryKeys: Map<String, Any>): String {
        return primaryKeys
            .map {
                when (val value = it.value) {
                    is String -> String.format("%s = '%s'", it.key, value)
                    is Boolean -> String.format("%s = '%s'", it.key, if (value) 1 else 0)
                    else -> String.format("%s = %s", it.key, it.value)
                }
            }
            .joinToString(separator = " and ")
    }
}