package com.amity.socialcloud.sdk.social.data.event

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.entity.social.event.EventEntity
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class EventLocalDataStore {

    fun hasInLocal(eventId: String): Boolean {
        return UserDatabase.get().eventDao().getByIdNow(eventId) != null
    }

    fun saveEvents(events: List<EventEntity>): Completable {
        return Completable.fromAction {
            UserDatabase.get().eventDao().save(events)
        }
    }

    fun observe(eventId: String): Flowable<EventEntity> {
        return UserDatabase.get().eventDao().observeEvent(eventId)
            .map {
                if (it.isEmpty()) {
                    throw AmityException.create(
                        message = "item not found",
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                } else {
                    return@map it[0]
                }
            }
    }

    fun getEvent(eventId: String): EventEntity? {
        var event: EventEntity? = null
        Completable.fromCallable {
            val entity = UserDatabase.get().eventDao().getByIdNow(eventId)
            if (entity != null) {
                event = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return event
    }

    fun getEvents(eventIds: List<String>): List<EventEntity> {
        var events = emptyList<EventEntity>()
        Completable.fromCallable {
            val eventDao = UserDatabase.get().eventDao()
            val entities = eventDao.getByIdsNow(eventIds)
            if (entities.isNotEmpty()) {
                events = entities.let { results ->
                    val orderById = eventIds.withIndex().associate { it.value to it.index }
                    val sortedEvents = results.sortedBy { orderById[it.eventId] }
                    sortedEvents
                }
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return events
    }

    fun getEventPagingSource(
        originType: AmityEventOriginType?,
        originId: String?,
        userId: String?,
        status: AmityEventStatus?,
        type: AmityEventType?,
        onlyAttendee: Boolean?,
        sortBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventSortOption,
        orderBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventOrderOption
    ): PagingSource<Int, EventEntity> {
        return UserDatabase.get().eventPagingDao().getEventPagingSource(
            originType = originType,
            originId = originId,
            userId = userId,
            status = status,
            type = type,
            onlyAttendee = onlyAttendee,
            sortBy = sortBy,
            orderBy = orderBy
        )
    }

    fun getLatestEvent(
        originType: AmityEventOriginType?,
        originId: String?,
        userId: String?,
        status: AmityEventStatus?,
        type: AmityEventType?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
        sortBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventSortOption,
        orderBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventOrderOption
    ): Flowable<EventEntity> {
        return UserDatabase.get().eventDao().getLatestEvent(
            originType = originType?.apiKey,
            originId = originId,
            userId = userId,
            status = status?.apiKey,
            type = type?.apiKey,
            hash = dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce = nonce,
            now = org.joda.time.DateTime.now(),
            sortBy = sortBy.value,
            orderBy = orderBy.value
        )
        //hash
    }

    fun notifyChanges(eventId: String) {
        UserDatabase.get().eventDao().notifyChanges(eventId)
    }
}
