package com.amity.socialcloud.sdk.social.data.event

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.entity.social.event.EventResponseEntity
import com.amity.socialcloud.sdk.model.social.event.AmityEventResponse
import com.amity.socialcloud.sdk.model.social.event.AmityEventResponseStatus
import com.amity.socialcloud.sdk.social.data.event.paging.EventResponseQueryMediator
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable

internal class AmityEventResponseRepository : AmityObjectRepository<EventResponseEntity, AmityEventResponse>() {

    override fun fetchAndSave(objectId: String): Completable {
        // Event response fetch is not supported directly by ID
        // They are fetched via event query APIs
        return Completable.complete()
    }

    override fun queryFromCache(objectId: String): EventResponseEntity? {
        return EventResponseLocalDataStore().getEventResponse(objectId)
    }

    override fun mapper(): ModelMapper<EventResponseEntity, AmityEventResponse> {
        return EventResponseModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EventResponseEntity> {
        return EventResponseLocalDataStore().observe(objectId)
    }

    @OptIn(ExperimentalPagingApi::class)
    fun getEventResponsePagingData(
        eventId: String,
        status: AmityEventResponseStatus?
    ): Flowable<PagingData<AmityEventResponse>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = 20,
                enablePlaceholders = true,
                initialLoadSize = 10,
                prefetchDistance = 0
            ),
            queryStreamMediator = EventResponseQueryMediator(
                eventId = eventId,
                status = status
            ),
            pagingSourceFactory = {
                EventResponseLocalDataStore().getEventResponsePagingSource(
                    eventId = eventId,
                    status = status
                )
            },
            modelMapper = EventResponseModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getLatestEventResponse(
        eventId: String,
        status: AmityEventResponseStatus?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<AmityEventResponse> {
        return EventResponseLocalDataStore().getLatestEventResponse(
            eventId = eventId,
            status = status,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).map { EventResponseModelMapper().map(it) }
    }
}
