package com.amity.socialcloud.sdk.social.data.event

import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventResponseStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.EventListDto
import com.ekoapp.ekosdk.internal.api.dto.EventRSVPListDto
import com.ekoapp.ekosdk.internal.api.socket.request.EventCreateRequest
import com.ekoapp.ekosdk.internal.api.socket.request.EventRSVPRequest
import com.ekoapp.ekosdk.internal.api.socket.request.EventUpdateRequest
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single
import org.joda.time.DateTime

internal class EventRemoteDataStore {

    fun getEvent(eventId: String): Single<EventListDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.getEvent(eventId)
        }
    }

    fun queryEvents(
        originType: AmityEventOriginType?,
        originId: String?,
        userId: String?,
        status: AmityEventStatus?,
        type: AmityEventType?,
        onlyAttendee: Boolean?,
        sortBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventSortOption,
        orderBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventOrderOption,
        limit: Int? = null,
        token: String? = null
    ): Single<EventListDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.queryEvents(
                originType = originType?.apiKey,
                originId = originId,
                userId = userId,
                status = status?.apiKey,
                type = type?.apiKey,
                onlyAttendee = onlyAttendee,
                sortBy = sortBy.value,
                orderBy = orderBy.value,
                limit = limit,
                token = token
            )
        }
    }

    fun createEvent(
        title: String?,
        description: String?,
        type: AmityEventType?,
        isInviteOnly: Boolean?,
        startTime: DateTime?,
        endTime: DateTime, // Made non-nullable (required)
        originType: AmityEventOriginType,
        originId: String,
        location: String?,
        externalUrl: String?,
        coverImageFileId: String?,
        tags: AmityTags?,
        timezone: String?
    ): Single<EventListDto> {
        val metadata = timezone?.let {
            JsonObject().apply {
                addProperty("timezone", it)
            }
        }

        val request = EventCreateRequest(
            title = title,
            description = description,
            type = type?.apiKey,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            originType = originType.apiKey,
            originId = originId,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags?.get(),
            metadata = metadata
        )

        return AmityHttpClient.get(EventApi::class).flatMap {
            it.createEvent(request)
        }
    }

    fun updateEvent(
        eventId: String,
        title: String?,
        description: String?,
        type: AmityEventType?,
        isInviteOnly: Boolean?,
        startTime: DateTime?,
        endTime: DateTime?,
        location: String?,
        externalUrl: String?,
        coverImageFileId: String?,
        tags: AmityTags?,
        timezone: String?
    ): Single<EventListDto> {
        val metadata = timezone?.let {
            JsonObject().apply {
                addProperty("timezone", it)
            }
        }

        val request = EventUpdateRequest(
            eventId = eventId,
            title = title,
            description = description,
            type = type?.apiKey,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags?.get(),
            metadata = metadata
        )

        return AmityHttpClient.get(EventApi::class).flatMap {
            it.updateEvent(eventId, request)
        }
    }

    fun deleteEvent(eventId: String): Single<com.ekoapp.ekosdk.internal.api.dto.EkoDeletionDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.deleteEvent(eventId)
        }
    }

    fun createRSVP(
        eventId: String,
        status: AmityEventResponseStatus
    ): Single<EventRSVPListDto> {
        val request = EventRSVPRequest(status = status.apiKey)
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.createRSVP(eventId, request)
        }
    }

    fun updateRSVP(
        eventId: String,
        status: AmityEventResponseStatus
    ): Single<EventRSVPListDto> {
        val request = EventRSVPRequest(status = status.apiKey)
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.updateRSVP(eventId, request)
        }
    }

    fun queryRSVPs(
        eventId: String,
        status: AmityEventResponseStatus?,
        limit: Int? = null,
        token: String? = null
    ): Single<EventRSVPListDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.queryRSVPs(
                eventId = eventId,
                status = status?.apiKey,
                limit = limit,
                token = token
            )
        }
    }

    fun getMyRSVP(eventId: String): Single<EventRSVPListDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.getMyRSVP(eventId)
        }
    }

}
