package com.amity.socialcloud.sdk.model.social.comment

import AmityPII
import android.os.Parcelable
import com.amity.socialcloud.sdk.api.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.helper.core.mention.AmityMentionee
import com.amity.socialcloud.sdk.model.core.events.AmityCommentEvents
import com.amity.socialcloud.sdk.model.core.events.AmityTopic
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionMap
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMember
import com.ekoapp.ekosdk.JsonArrayParceler
import com.ekoapp.ekosdk.ReactorObject
import com.ekoapp.ekosdk.internal.data.converter.EkoGson
import com.ekoapp.ekosdk.internal.util.PIIUtils
import com.google.common.base.Objects
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.amity.types.ObjectId
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject, JsonObjectParceler>
@TypeParceler<JsonObject?, JsonObjectParceler>
@TypeParceler<JsonArray, JsonArrayParceler>
data class AmityComment internal constructor(
    private var commentId: String = ObjectId.get().toHexString(),
    internal var referenceType: String,
    internal var referenceId: String,
    private var userId: String,
    private var parentId: String?,
    private var rootId: String?,
    private var dataType: String = DataType.TEXT.apiKey,
    private var dataTypes: List<String> = emptyList(),
    private var data: JsonObject,
    internal var rawAttachments: JsonArray,
    internal var attachments: List<Attachment> = emptyList(),
    private var metadata: JsonObject?, // Remain private until further requirement change
    private var childrenNumber: Int,
    internal var latestReplies: List<AmityComment> = emptyList(),
    private var flagCount: Int,
    internal var myReactions: List<String>,
    private var reactionCount: Int,
    private var reactions: AmityReactionMap,
    private var isDeleted: Boolean,
    private var editedAt: DateTime,
    private var createdAt: DateTime,
    private var updatedAt: DateTime,
    private var syncState: String,
    internal var mentionees: List<AmityMentionee>,
    internal var user: AmityUser?,
    internal var isFlaggedByMe: Boolean = false,
    internal val path: String,
    internal val target: Target,
    internal var piiList: List<AmityPII> = emptyList(),
) : Parcelable, ReactorObject {

    fun getCommentId(): String {
        return commentId
    }

    fun getReference(): Reference {
        return Reference.from(referenceType, referenceId)
    }

    fun getParentId(): String? {
        return parentId
    }

    fun getCreatorId(): String {
        return userId
    }

    fun getState(): State {
        return State.enumOf(syncState)
    }

    fun getDataTypes(): Set<DataType> {
        return dataTypes.map { DataType.enumOf(it) }.toSet()
    }

    fun getData(): Data {
        return Data.from(data, piiList, metadata)
    }

    fun getAttachments(): List<Attachment> {
        return attachments
    }

    fun getChildCount(): Int {
        return childrenNumber
    }

    fun getFlagCount(): Int {
        return flagCount
    }

    fun getReactionCount(): Int {
        return reactionCount
    }

    fun getReactionMap(): AmityReactionMap {
        return reactions
    }

    fun getMyReactions(): List<String> {
        return myReactions
    }

    fun getLatestReplies(): List<AmityComment> {
        return latestReplies
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getEditedAt(): DateTime {
        return editedAt
    }

    fun getCreatedAt(): DateTime {
        return createdAt
    }

    fun getUpdatedAt(): DateTime {
        return updatedAt
    }

    fun getCreator(): AmityUser? {
        return user
    }

    fun isEdited(): Boolean {
        return editedAt.isAfter(createdAt)
    }

    /* begin_public_function
        id: comment.check_flag_by_me
        api_style: async
    */
    fun isFlaggedByMe(): Boolean {
        return isFlaggedByMe
    }
    /* end_public_function */

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getMentionees(): List<AmityMentionee> {
        return mentionees
    }

    fun getTarget(): Target {
        return target
    }

    /**
     * Returns a list of PII entities detected in the comment's content.
     * Returns an empty list if no PII was detected.
     */
    fun getPIIData(): List<AmityPII> {
        return when (val commentData = getData()) {
            is Data.TEXT -> commentData.piiList
            else -> emptyList()
        }
    }

    sealed class Reference {

        class POST(private val postId: String) : Reference() {
            fun getPostId(): String {
                return postId
            }
        }

        class STORY(private val storyId: String) : Reference() {
            fun getStoryId(): String {
                return storyId
            }
        }

        class CONTENT(private val contentId: String) : Reference() {
            fun getContentId(): String {
                return contentId
            }
        }

        object UNKNOWN : Reference()

        companion object {
            internal fun from(referenceType: String, referenceId: String): Reference {
                return when (referenceType) {
                    "post" -> {
                        POST(referenceId)
                    }

                    "content" -> {
                        CONTENT(referenceId)
                    }

                    "story" -> {
                        STORY(referenceId)
                    }

                    else -> {
                        UNKNOWN
                    }
                }
            }
        }
    }

    sealed class Data : Parcelable {

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class TEXT(
            internal val commentId: String,
            internal val text: String? = "",
            internal val piiList: List<AmityPII> = emptyList(),
            internal val mentionJson: JsonObject? = null,
        ) : Data() {

            fun getCommentId(): String {
                return commentId
            }

            fun getText(): String {
                return text ?: ""
            }

            /**
             * Returns a new TEXT object with PII entities in the text redacted.
             * This method is non-destructive and does not alter the original object.
             *
             * @param piiCategories A list of PII categories to redact. Defaults to empty list (redacts all categories).
             * @param replaceChar The character used to replace PII. Defaults to '*'.
             * @return A redacted version of the text.
             */
            fun redactedText(
                piiCategories: List<AmityPII.Category> = emptyList(),
                replaceChar: Char = '*'
            ): String {
                return PIIUtils.redactPIIFromText(
                    originalText = this.text ?: "",
                    piiList = this.piiList,
                    mentionJson = this.mentionJson,
                    piiCategories = piiCategories,
                    replaceChar = replaceChar
                )
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is TEXT
                        && Objects.equal(other.commentId, commentId)
                        && Objects.equal(other.text, text)
                        && Objects.equal(other.piiList, piiList)
                        && Objects.equal(other.mentionJson, mentionJson))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(commentId, text, piiList, mentionJson)
            }
        }

        companion object {
            internal fun from(data: JsonObject): Data {
                return EkoGson.get().fromJson(data, TEXT::class.java)
            }

            internal fun from(
                data: JsonObject,
                piiList: List<AmityPII> = emptyList(),
                metadata: JsonObject? = null
            ): Data {
                val textData = EkoGson.get().fromJson(data, TEXT::class.java)
                return TEXT(
                    commentId = textData.commentId,
                    text = textData.text,
                    piiList = piiList,
                    mentionJson = metadata
                )
            }
        }

    }

    sealed class Attachment : Parcelable {

        abstract fun getDataType(): DataType

        @Parcelize
        class IMAGE(
            private val fileId: String,
            private val image: AmityImage?
        ) : Attachment() {

            override fun getDataType(): DataType {
                return DataType.IMAGE
            }

            fun getFileId(): String {
                return fileId
            }

            fun getImage(): AmityImage? {
                return image
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is IMAGE
                        && Objects.equal(other.fileId, fileId))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(fileId, getDataType())
            }
        }
    }

    enum class DataType(val apiKey: String) {
        TEXT("text"),
        IMAGE("image"),
        UNKNOWN("unknown");

        companion object {
            fun enumOf(value: String): DataType {
                return values().find { it.apiKey == value } ?: UNKNOWN
            }
        }
    }

    internal enum class TargetType(val apiKey: String) {
        COMMUNITY("community"),
        USER("user"),
        CONTENT("content"),
        UNKNOWN("unknown");

        companion object {
            fun enumOf(value: String?): TargetType {
                return values().find { it.apiKey == value } ?: UNKNOWN
            }
        }
    }

    sealed class Target(
        private val targetType: TargetType,
        private val targetId: String
    ) : Parcelable {

        internal fun getTargetType(): TargetType {
            return targetType
        }

        internal fun getTargetId(): String {
            return targetId
        }

        @Parcelize
        class COMMUNITY internal constructor(
            internal val targetCommunityId: String,
            internal val creatorCommunityMemberId: String,
            internal var creatorCommunityMember: AmityCommunityMember? = null,
        ) : Target(TargetType.COMMUNITY, targetCommunityId) {

            fun getCommunityId(): String {
                return targetCommunityId
            }

            fun getCreatorMember(): AmityCommunityMember? {
                return creatorCommunityMember
            }
        }

        @Parcelize
        class USER internal constructor(
            private val targetUserId: String,
        ) : Target(TargetType.USER, targetUserId) {

            fun getUserId(): String {
                return targetUserId
            }
        }

        @Parcelize
        class CONTENT internal constructor(
            private val targetContentId: String,
        ) : Target(TargetType.CONTENT, targetContentId) {

            fun getContentId(): String {
                return targetContentId
            }
        }

        @Parcelize
        object UNKNOWN : Target(TargetType.UNKNOWN, "")
    }

    enum class State(val stateName: String) {
        CREATED("created"),
        SYNCING("syncing"),
        SYNCED("synced"),
        FAILED("failed");

        companion object {
            fun enumOf(value: String): State = values().find { it.stateName == value } ?: SYNCED
        }
    }

    fun subscription(events: AmityCommentEvents): AmityTopicSubscription {
        return AmityTopicSubscription(AmityTopic.COMMENT(this, events))
    }

    override fun uniqueId(): String {
        return commentId
    }

    override fun updatedAt(): DateTime {
        return updatedAt
    }

}