package com.amity.socialcloud.sdk.social.data.event.paging

import androidx.paging.PagingState
import androidx.paging.rxjava3.RxPagingSource
import com.amity.socialcloud.sdk.model.social.event.AmityEventResponse
import com.amity.socialcloud.sdk.model.social.event.AmityEventResponseStatus
import com.amity.socialcloud.sdk.social.data.event.EventRemoteDataStore
import com.amity.socialcloud.sdk.social.data.event.EventResponseEntityMapper
import com.amity.socialcloud.sdk.social.data.event.EventResponseModelMapper
import com.amity.socialcloud.sdk.social.data.event.EventResponsePersister
import com.amity.socialcloud.sdk.social.domain.event.EventComposerUseCase
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

internal class EventRSVPPagingSource(
    private val eventId: String,
    private val status: AmityEventResponseStatus?
) : RxPagingSource<String, AmityEventResponse>() {

    override fun loadSingle(params: LoadParams<String>): Single<LoadResult<String, AmityEventResponse>> {
        return EventRemoteDataStore()
            .queryRSVPs(
                eventId = eventId,
                status = status,
                limit = params.loadSize,
                token = params.key
            )
            .subscribeOn(Schedulers.io())
            .flatMap { dto ->
                EventResponsePersister().persist(dto)
                    .andThen(Single.fromCallable {
                        val responseEntities = dto.eventResponses
                            ?.let { EventResponseEntityMapper().map(it) }
                            ?: emptyList()

                        val rsvps = responseEntities.map { EventResponseModelMapper().map(it) }

                        // Compose each RSVP with linked objects
                        rsvps.forEach { rsvp ->
                            EventComposerUseCase()
                                .composeRSVP(rsvp, dto.events, dto.users, dto.files)
                                .blockingAwait()
                        }

                        LoadResult.Page(
                            data = rsvps,
                            prevKey = null,
                            nextKey = dto.paging?.next
                        ) as LoadResult<String, AmityEventResponse>
                    })
            }
            .onErrorReturn { error ->
                LoadResult.Error(error)
            }
    }

    override fun getRefreshKey(state: PagingState<String, AmityEventResponse>): String? {
        return null
    }
}
