package com.amity.socialcloud.sdk.social.data.event.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.api.social.event.query.AmityEventOrderOption
import com.amity.socialcloud.sdk.api.social.event.query.AmityEventSortOption
import com.amity.socialcloud.sdk.entity.social.event.EventEntity
import com.amity.socialcloud.sdk.model.social.event.AmityEvent
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.amity.socialcloud.sdk.social.data.event.EventQueryPersister
import com.amity.socialcloud.sdk.social.data.event.EventRemoteDataStore
import com.amity.socialcloud.sdk.social.data.event.EventRepository
import com.ekoapp.ekosdk.internal.api.dto.EventListDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@ExperimentalPagingApi
internal class EventQueryMediator(
    private val originType: AmityEventOriginType?,
    private val originId: String?,
    private val userId: String?,
    private val status: AmityEventStatus?,
    private val type: AmityEventType?,
    private val onlyAttendee: Boolean?,
    private val sortBy: AmityEventSortOption,
    private val orderBy: AmityEventOrderOption
) : DynamicQueryStreamMediator<EventEntity, EventListDto, AmityEvent>(
    nonce = AmityNonce.EVENT_QUERY,
    dynamicQueryStreamKeyCreator = EventQueryKeyCreator(
        originType = originType,
        originId = originId,
        userId = userId,
        status = status,
        type = type,
        onlyAttendee = onlyAttendee,
        sortBy = sortBy,
        orderBy = orderBy
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityEvent> {
        return EventRepository().getLatestEvent(
            originType = originType,
            originId = originId,
            userId = userId,
            status = status,
            type = type,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce,
            sortBy = sortBy,
            orderBy = orderBy
        ).toRx2()
        // on next
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EventListDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<EventListDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EventListDto): Completable {
        return EventQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EventListDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.events?.mapNotNull { it.eventId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EventListDto> {
        return EventRemoteDataStore().queryEvents(
            originType = originType,
            originId = originId,
            userId = userId,
            status = status,
            type = type,
            onlyAttendee = onlyAttendee,
            sortBy = sortBy,
            orderBy = orderBy,
            limit = limit,
            token = token
        ).toRx2()
    }
}
