package com.amity.socialcloud.sdk.core.session.component

import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionLifeCycle
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.log.AmityLog
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.schedulers.Schedulers

abstract class SessionComponent(
    private val sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    private val sessionStateEventBus: SessionStateEventBus
) {

    var sessionState: SessionState = SessionState.NotLoggedIn
        private set

    abstract fun onSessionStateChange(sessionState: SessionState)

    abstract fun establish(account: EkoAccount)

    abstract fun destroy()

    abstract fun handleTokenExpire()

    fun getSessionStateEventBus(): SessionStateEventBus {
        return sessionStateEventBus
    }

    fun getSessionLifeCycleEventBus(): SessionLifeCycleEventBus {
        return sessionLifeCycleEventBus
    }

    init {
        initSessionState()
        observeSessionLifeCycle()
        observeSessionState()
    }
    
    private fun initSessionState() {
        sessionStateEventBus.getCurrentEvent()?.let {
            sessionState = it
        }
    }

    private fun observeSessionLifeCycle() {
        sessionLifeCycleEventBus
            .observe()
            .distinctUntilChanged()
            .subscribeOn(Schedulers.io())
            .doOnNext { sessionComponentEvent ->
                when (sessionComponentEvent) {
                    is SessionLifeCycle.Establish -> {
                        establish(sessionComponentEvent.account)
                    }
                    is SessionLifeCycle.Destroy -> {
                        destroy()
                    }
                    is SessionLifeCycle.HandleTokenExpire -> {
                        handleTokenExpire()
                    }
                    is SessionLifeCycle.StateChange -> {
                        onSessionStateChange(sessionComponentEvent.sessionState)
                        this.sessionState = sessionComponentEvent.sessionState
                    }
                }
            }
            .subscribe()
    }


    private fun observeSessionState() {
        sessionStateEventBus
            .observe()
            .distinctUntilChanged()
            .subscribeOn(Schedulers.io())
            .doOnNext { sessionState ->
                onSessionStateChange(sessionState)
                this.sessionState = sessionState
            }
            .subscribe()
    }
}