package com.amity.socialcloud.sdk.social.data.post

import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.data.mention.LegacyMentioneeModelMapper
import com.amity.socialcloud.sdk.model.core.file.AmityClip
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.util.PIIUtils
import com.google.gson.JsonObject

internal class PostModelMapper : ModelMapper<PostEntity, AmityPost>() {

    override fun map(entity: PostEntity): AmityPost {

        return entity.run {

            AmityPost(
                postId = postId,
                mid = mid,
                target = when (AmityPost.TargetType.enumOf(targetType)) {
                    AmityPost.TargetType.COMMUNITY -> AmityPost.Target.COMMUNITY(
                        targetId,
                        postedUserId
                    )

                    AmityPost.TargetType.USER -> AmityPost.Target.USER(targetId)
                },
                parentPostId = parentPostId,
                postedUserId = postedUserId,
                sharedUserId = sharedUserId,
                structureType = structureType,
                type = AmityPost.DataType.sealedOf(postDataType),
                data = when (AmityPost.DataType.sealedOf(postDataType)) {
                    AmityPost.DataType.TEXT -> AmityPost.Data.TEXT(
                        postId = postId,
                        text = serializeText(data),
                        title = serializeTitle(data),
                        mentionJson = metadata,
                        piiList = PIIUtils.parsePIIData(pii),
                    )

                    AmityPost.DataType.IMAGE -> AmityPost.Data.IMAGE(
                        postId,
                        serializeImageFileId(data),
                        data
                    )

                    AmityPost.DataType.FILE -> AmityPost.Data.FILE(
                        postId,
                        serializeFileId(data),
                        data
                    )

                    AmityPost.DataType.AUDIO -> AmityPost.Data.AUDIO(
                        postId,
                        serializeFileId(data),
                        data
                    )

                    AmityPost.DataType.VIDEO -> AmityPost.Data.VIDEO(
                        postId,
                        serializeThumbnailFileId(data),
                        data
                    )

                    AmityPost.DataType.CLIP -> AmityPost.Data.CLIP(
                        postId = postId,
                        thumbnailFileId = serializeThumbnailFileId(data),
                        rawData = data,
                        fileId = serializeFileId(data),
                        isMuted = serializeIsMute(data),
                        displayMode = AmityClip.DisplayMode.enumOf(serializeDisplayMode(data))
                    )

                    AmityPost.DataType.LIVE_STREAM -> AmityPost.Data.LIVE_STREAM(
                        postId,
                        serializeLiveStreamId(data),
                        data
                    )

                    AmityPost.DataType.POLL -> AmityPost.Data.POLL(
                        postId,
                        serializePollId(data),
                        data
                    )

                    AmityPost.DataType.ROOM -> AmityPost.Data.ROOM(
                            postId,
                            serializeRoomId(data),
                            data
                    )

                    else -> AmityPost.Data.CUSTOM(
                        postId,
                        AmityPost.DataType.CUSTOM(postDataType),
                        data
                    )
                },
                metadata = metadata,
                sharedCount = sharedCount,
                reactions = reactions,
                reactionCount = reactionCount,
                commentCount = commentCount,
                flagCount = flagCount,
                childrenPostIds = childPostIds,
                isDeleted = isDeleted,
                feedType = AmityFeedType.enumOf(feedType),
                editedAt = editedAt,
                createdAt = createdAt,
                updatedAt = updatedAt,
                mentionees = LegacyMentioneeModelMapper().map(mentionees),
                path = path,
                impression = impression,
                reach = reach,
                hashtags = hashtags,
            )
        }
    }

    private fun serializeText(rawData: JsonObject?): String {
        return rawData?.get("text")?.asString ?: ""
    }

    private fun serializeTitle(rawData: JsonObject?): String {
        return rawData?.get("title")?.asString ?: ""
    }

    private fun serializeImageFileId(rawData: JsonObject?): String {
        return rawData?.get("fileId")?.asString ?: ""
    }

    private fun serializeFileId(rawData: JsonObject?): String {
        return rawData?.get("fileId")?.asString ?: ""
    }

    private fun serializeThumbnailFileId(rawData: JsonObject?): String {
        return rawData?.get("thumbnailFileId")?.asString ?: ""
    }

    private fun serializeLiveStreamId(rawData: JsonObject?): String {
        return rawData?.get("streamId")?.asString ?: ""
    }

    private fun serializeIsMute(rawData: JsonObject?): Boolean {
        return rawData?.get("isMuted")?.asBoolean == true
    }

    private fun serializeDisplayMode(rawData: JsonObject?): String {
        return rawData?.get("displayMode")?.asString ?: ""
    }

    private fun serializePollId(rawData: JsonObject?): String {
        return rawData?.get("pollId")?.asString ?: ""
    }

    private fun serializeRoomId(rawData: JsonObject?): String {
        return rawData?.get("roomId")?.asString ?: ""
    }

}