package com.ekoapp.ekosdk.internal.util

import AmityPII
import com.amity.socialcloud.sdk.helper.core.mention.AmityMentionMetadataGetter
import com.ekoapp.ekosdk.internal.api.dto.EkoPiiDto
import com.google.gson.JsonArray
import com.google.gson.JsonObject

/**
 * Utility class for handling PII data serialization, deserialization, and redaction
 */
internal object PIIUtils {

    /**
     * Converts JsonArray directly to a list of AmityPII objects
     */
    fun parsePIIData(piiArray: JsonArray?): List<AmityPII> {
        if (piiArray == null) return emptyList()
        return parsePIIDataFromArray(piiArray)
    }

    /**
     * Converts a list of EkoPiiDto objects to JsonArray for entity storage
     * This is used by EntityMappers to convert DTO objects to JSON for database storage
     */
    fun convertDtosToJsonArray(piiDtos: List<EkoPiiDto>): JsonArray {
        val piiArray = JsonArray()
        piiDtos.forEach { piiDto ->
            val piiObject = JsonObject().apply {
                addProperty("category", piiDto.category)
                addProperty("offset", piiDto.offset)
                addProperty("length", piiDto.length)
                addProperty("confidenceScore", piiDto.confidenceScore)
            }
            piiArray.add(piiObject)
        }
        return piiArray
    }

    /**
     * Internal function to parse PII data from JsonArray
     */
    private fun parsePIIDataFromArray(piiArray: JsonArray): List<AmityPII> {
        return try {
            piiArray.mapNotNull { element ->
                val piiObject = element.asJsonObject
                val category = piiObject.get("category")?.asString ?: return@mapNotNull null
                val offset = piiObject.get("offset")?.asInt ?: return@mapNotNull null
                val length = piiObject.get("length")?.asInt ?: return@mapNotNull null
                val confidence = piiObject.get("confidenceScore")?.asDouble ?: 0.0
                
                val categoryEnum = parseCategoryFromString(category)
                AmityPII(offset, length, confidence, categoryEnum)
            }
        } catch (e: Exception) {
            // Log error in production
            emptyList()
        }
    }

    /**
     * Converts string representation to AmityPII.Category
     */
    private fun parseCategoryFromString(categoryString: String): AmityPII.Category {
        return when (categoryString) {
            "Email" -> AmityPII.Category.EMAIL
            "PhoneNumber" -> AmityPII.Category.PHONE_NUMBER
            "IPAddress" -> AmityPII.Category.IP_ADDRESS
            "Address" -> AmityPII.Category.ADDRESS
            "PassportNumber" -> AmityPII.Category.PASSPORT_NUMBER
            else -> AmityPII.Category.OTHERS(categoryString)
        }
    }

    /**
     * Redacts PII from text while preserving mentions
     * This is a centralized implementation used by AmityPost, AmityComment, and AmityMessage
     *
     * @param originalText The original text content
     * @param piiList List of PII entities detected in the text
     * @param mentionJson JSON object containing mention metadata
     * @param piiCategories List of PII categories to redact (empty means redact all)
     * @param replaceChar Character to use for redaction (default: '*')
     * @return Redacted text with PII replaced but mentions preserved
     */
    fun redactPIIFromText(
        originalText: String,
        piiList: List<AmityPII>,
        mentionJson: JsonObject?,
        piiCategories: List<AmityPII.Category> = emptyList(),
        replaceChar: Char = '*'
    ): String {
        // Use a mutable data structure for efficient character replacement
        val mutableText = originalText.toCharArray()
        
        // Sort PII by offset to handle replacement correctly
        val sortedPIIList = piiList.sortedBy { it.offset }
        
        // Extract mention ranges from metadata
        val mentionRangesList = extractMentionRangesFromMetadata(mentionJson)
        // Iterate through each detected PII entity
        for (pii in sortedPIIList) {
            // 1. Check if the category should be redacted
            // If piiCategories is empty, redact all categories
            // Otherwise, only redact if the category is in the list
            if (piiCategories.isNotEmpty() && pii.category !in piiCategories) {
                continue // Skip to the next PII item
            }
            
            // 2. Define the range for the current PII item
            val piiStart = pii.offset
            val piiEnd = pii.offset + pii.length
            
            // 3. Check for overlaps with mentions
            var isOverlapping = false
            for (mentionRange in mentionRangesList) {
                if (rangesOverlap(piiStart, piiEnd, mentionRange.first, mentionRange.second)) {
                    isOverlapping = true
                    break
                }
            }
            if (isOverlapping) continue
            
            // 4. Perform the replacement if no overlaps found
            for (i in piiStart until piiEnd) {
                if (i < mutableText.size) {
                    mutableText[i] = replaceChar
                }
            }
        }
        
        // Convert character array back to a string and return
        return String(mutableText)
    }

    /**
     * Extracts mention ranges from JSON metadata
     * Handles both user and channel mentions
     *
     * @param mentionJson JSON object containing mention metadata
     * @return List of mention ranges as Pair<start, end>
     */
    fun extractMentionRangesFromMetadata(mentionJson: JsonObject?): List<Pair<Int, Int>> {
        if (mentionJson == null) {
            return emptyList()
        }
        
        return try {
            val mentionMetadataGetter = AmityMentionMetadataGetter(mentionJson)
            val ranges = mutableListOf<Pair<Int, Int>>()
            
            // Extract user mention ranges
            mentionMetadataGetter.getMentionedUsers().forEach { userMention ->
                val startIndex = userMention.getIndex()
                val length = userMention.getLength()
                if (startIndex >= 0 && length > 0) {
                    ranges.add(Pair(startIndex, startIndex + length))
                }
            }
            
            // Extract channel mention ranges
            mentionMetadataGetter.getMentionedChannels().forEach { channelMention ->
                val startIndex = channelMention.getIndex()
                val length = channelMention.getLength()
                if (startIndex >= 0 && length > 0) {
                    ranges.add(Pair(startIndex, startIndex + length))
                }
            }
            
            ranges
        } catch (e: Exception) {
            // Fallback to empty list if there's any issue parsing the mention data
            emptyList()
        }
    }

    /**
     * Checks if two ranges overlap
     *
     * @param start1 Start of first range
     * @param end1 End of first range
     * @param start2 Start of second range
     * @param end2 End of second range
     * @return true if ranges overlap, false otherwise
     */
    private fun rangesOverlap(start1: Int, end1: Int, start2: Int, end2: Int): Boolean {
        return start1 < end2 && start2 < end1
    }
}
