package com.amity.socialcloud.sdk.api.social.post

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.social.post.create.target.AmityPostCreationTarget
import com.amity.socialcloud.sdk.api.social.post.query.AmityPostQueryTarget
import com.amity.socialcloud.sdk.api.social.post.update.AmityPostUpdate
import com.amity.socialcloud.sdk.helper.core.hashtag.AmityHashtag
import com.amity.socialcloud.sdk.helper.core.hashtag.AmityHashtagBuilder
import com.amity.socialcloud.sdk.model.core.file.AmityClip
import com.amity.socialcloud.sdk.model.core.file.AmityFile
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.flag.AmityContentFlagReason
import com.amity.socialcloud.sdk.model.core.pin.AmityPinSortByOptions
import com.amity.socialcloud.sdk.model.core.pin.AmityPinnedPost
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.domain.post.SemanticSearchPostUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.ClipPostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.CustomPostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.FilePostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.ImagePostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.LiveStreamPostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.PollPostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.RoomPostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.TextPostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.create.VideoPostCreateUseCase
import com.amity.socialcloud.sdk.social.domain.post.getbyids.PostGetByIdsUseCase
import com.amity.socialcloud.sdk.social.domain.post.update.PostEditUseCase
import com.ekoapp.ekosdk.internal.api.socket.request.FlagContentRequest
import com.ekoapp.ekosdk.internal.usecase.post.FlagPostUseCase
import com.ekoapp.ekosdk.internal.usecase.post.GlobalPinnedPostQueryUseCase
import com.ekoapp.ekosdk.internal.usecase.post.PinnedPostQueryUseCase
import com.ekoapp.ekosdk.internal.usecase.post.PostApproveUseCase
import com.ekoapp.ekosdk.internal.usecase.post.PostDeclineUseCase
import com.ekoapp.ekosdk.internal.usecase.post.PostDeleteUseCase
import com.ekoapp.ekosdk.internal.usecase.post.PostLiveObjectUseCase
import com.ekoapp.ekosdk.internal.usecase.post.SearchPostsByHashtagUseCase
import com.ekoapp.ekosdk.internal.usecase.post.UnFlagPostUseCase
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

class AmityPostRepository internal constructor() {

    /* begin_public_function
	  id: post.get
	*/
    /**
     * Gets a LiveObject of post
     * @param [postId] a postId
     * @return LiveObject of post as Flowable<AmityPost>
     */
    fun getPost(postId: String): Flowable<AmityPost> {
        return PostLiveObjectUseCase().execute(postId)
    }
    /* end_public_function */

    /* begin_public_function
      id: post.get_by_ids
	  api_style: live_collection
   */
    /**
     * Gets a LiveCollection of posts
     * @param [postIds] an array of postId
     * @return LiveCollection of post as Flowable<List<AmityPost>>
     */
    fun getPostByIds(postIds: Set<String>): Flowable<List<AmityPost>> {
        return PostGetByIdsUseCase().execute(postIds)
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.query
	  api_style: live_collection
	*/
    /**
     * Initiates a function chain to query for post.
     * @return new instance of AmityPostQueryTarget.
     */
    fun getPosts(): AmityPostQueryTarget {
        return AmityPostQueryTarget()
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.queryByHashtag
	  api_style: live_collection
	*/
    /**
     * Search for posts that contain specific hashtags.
     * @param [hashtags] List of hashtags to search for (without the # symbol)
     * @param [dataTypes] Types of posts to include in the search results (default: all types)
     * @return LiveCollection of posts as Flowable<PagingData<AmityPost>>
     */
    fun searchPostsByHashtag(
        hashtags: List<String>,
        dataTypes: List<AmityPost.DataType> = listOf(),
    ): Flowable<PagingData<AmityPost>> {
        return SearchPostsByHashtagUseCase().execute(
            hashtags = hashtags,
            dataTypes = dataTypes,
        )
    }
    /* end_public_function */


    /* begin_public_function
	  id: post.create.text_post
	  api_style: async
	*/
    /**
     * create a text post
     * @return Async post as Single<AmityPost>
     */
    fun createTextPost(
        targetType: AmityPost.TargetType,
        targetId: String,
        text: String,
        title: String? = null,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null
    ): Single<AmityPost> {
        return TextPostCreateUseCase().execute(
            targetType = targetType.apiKey,
            targetId = targetId,
            text = text,
            title = title,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    @Deprecated("Deprecated AmityPost.Target")
    fun createTextPost(
        target: AmityPost.Target,
        text: String,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return TextPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            text = text,
            title = null,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }

    /* begin_public_function
	  id: post.create.image_post
	  api_style: async
	*/
    /**
     * create an image post
     * @return Async post as Single<AmityPost>
     */
    fun createImagePost(
        targetType: AmityPost.TargetType,
        targetId: String,
        images: Set<AmityImage>,
        text: String = "",
        title: String? = null,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return ImagePostCreateUseCase().execute(
            targetType = targetType.apiKey,
            targetId = targetId,
            images = images.toList(),
            text = text,
            title = title,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    @Deprecated("Deprecated AmityPost.Target")
    fun createImagePost(
        target: AmityPost.Target,
        images: Set<AmityImage>,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return ImagePostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            images = images.toList(),
            text = text,
            title = null,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }

    /* begin_public_function
	  id: post.create.file_post
	  api_style: async
	*/
    /**
     * create a file post
     * @return Async post as Single<AmityPost>
     */
    fun createFilePost(
        targetType: AmityPost.TargetType,
        targetId: String,
        files: Set<AmityFile>,
        text: String = "",
        title: String? = null,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return FilePostCreateUseCase().execute(
            targetType = targetType.apiKey,
            targetId = targetId,
            files = files.toList(),
            text = text,
            title = title,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    @Deprecated("Deprecated AmityPost.Target")
    fun createFilePost(
        target: AmityPost.Target,
        files: Set<AmityFile>,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return FilePostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            files = files.toList(),
            text = text,
            title = null,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }

    /* begin_public_function
	  id: post.create.video_post
	  api_style: async
	*/
    /**
     * create a video post
     * @return Async post as Single<AmityPost>
     */
    fun createVideoPost(
        targetType: AmityPost.TargetType,
        targetId: String,
        videos: Set<AmityVideo>,
        text: String = "",
        title: String? = null,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return VideoPostCreateUseCase().execute(
            targetType = targetType.apiKey,
            targetId = targetId,
            videos = videos.toList(),
            text = text,
            title = title,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    @Deprecated("Deprecated AmityPost.Target")
    fun createVideoPost(
        target: AmityPost.Target,
        videos: Set<AmityVideo>,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return VideoPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            videos = videos.toList(),
            text = text,
            title = null,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }

    /* begin_public_function
       id: post.create.clip_post
       api_style: async
    */
    /**
     * create a video post
     * @return Async post as Single<AmityPost>
     */
    fun createClipPost(
        targetType: AmityPost.TargetType,
        targetId: String,
        clip: AmityClip,
        text: String = "",
        displayMode: AmityClip.DisplayMode,
        isMuted: Boolean = false,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return ClipPostCreateUseCase().execute(
            targetType = targetType.apiKey,
            targetId = targetId,
            clip = clip,
            text = text,
            displayMode = displayMode,
            isMuted = isMuted,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.create.poll_post
	  api_style: async
	*/
    /**
     * create a poll post
     * @return Async post as Single<AmityPost>
     */
    fun createPollPost(
        targetType: AmityPost.TargetType,
        targetId: String,
        pollId: String,
        text: String = "",
        title: String? = null,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return PollPostCreateUseCase().execute(
            targetType = targetType.apiKey,
            targetId = targetId,
            pollId = pollId,
            text = text,
            title = title,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    @Deprecated("Deprecated AmityPost.Target")
    fun createPollPost(
        target: AmityPost.Target,
        pollId: String,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return PollPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            pollId = pollId,
            text = text,
            metadata = metadata,
            title = null,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }

    /* begin_public_function
	  id: post.create.livestream_post
	  api_style: async
	*/
    /**
     * create a live stream post
     * @return Async post as Single<AmityPost>
     */
    fun createLiveStreamPost(
        targetType: AmityPost.TargetType,
        targetId: String,
        streamId: String,
        text: String = "",
        title: String? = null,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return LiveStreamPostCreateUseCase().execute(
            targetType = targetType.apiKey,
            targetId = targetId,
            streamId = streamId,
            text = text,
            title = title,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.create.livestream_post
	  api_style: async
	*/
    /**
     * create a live stream post
     * @return Async post as Single<AmityPost>
     */
    fun createRoomPost(
        targetType: AmityPost.TargetType,
        targetId: String,
        streamId: String,
        text: String = "",
        title: String? = null,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return RoomPostCreateUseCase().execute(
                targetType = targetType.apiKey,
                targetId = targetId,
                roomId = streamId,
                text = text,
                title = title,
                metadata = metadata,
                mentionUserIds = mentionUserIds?.toList(),
                hashtags = hashtags,
        )
    }
    /* end_public_function */

    @Deprecated("Deprecated AmityPost.Target")
    fun createLiveStreamPost(
        target: AmityPost.Target,
        streamId: String,
        text: String = "",
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return LiveStreamPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            streamId = streamId,
            text = text,
            title = null,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }

    /* begin_public_function
	  id: post.create.custom_post
	  api_style: async
	*/
    /**
     * create a custom post
     * @return Async post as Single<AmityPost>
     */
    fun createCustomPost(
        targetType: AmityPost.TargetType,
        targetId: String,
        customDataType: String,
        data: JsonObject,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return CustomPostCreateUseCase().execute(
            targetType = targetType.apiKey,
            targetId = targetId,
            dataType = AmityPost.DataType.CUSTOM(customDataType),
            data = data,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    @Deprecated("Deprecated AmityPost.Target")
    fun createCustomPost(
        target: AmityPost.Target,
        customDataType: String,
        data: JsonObject,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Single<AmityPost> {
        return CustomPostCreateUseCase().execute(
            targetType = target.getTargetType().apiKey,
            targetId = target.getTargetId(),
            dataType = AmityPost.DataType.CUSTOM(customDataType),
            data = data,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }

    /* begin_public_function
	  id: post.edit
	  api_style: async
	*/
    /**
     * Initiates a function chain to edit a post.
     * @param [postId] a postId
     * @return new instance of AmityPostUpdate.Builder.
     */
    fun editPost(postId: String): AmityPostUpdate.Builder {
        return AmityPostUpdate.Builder(postId)
    }
    /* end_public_function */

    /* begin_public_function
     id: post.edit.custom_post
     api_style: async
   */
    /**
     * edit a custom post
     * @param [postId] a postId
     * @return Async as Completable
     */
    fun editCustomPost(
        postId: String,
        data: JsonObject,
        metadata: JsonObject? = null,
        mentionUserIds: Set<String>? = null,
        hashtags: List<String>? = null,
    ): Completable {
        return PostEditUseCase().execute(
            postId = postId,
            data = data,
            metadata = metadata,
            mentionUserIds = mentionUserIds?.toList(),
            hashtags = hashtags,
        )
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.soft_delete
	*/
    /**
     * soft-deletes a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun softDeletePost(postId: String): Completable {
        return PostDeleteUseCase().execute(postId, false)
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.hard_delete
	*/
    /**
     * hard-deletes a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun hardDeletePost(postId: String): Completable {
        return PostDeleteUseCase().execute(postId, true)
    }
    /* end_public_function */


    /* begin_public_function
	  id: post.approve
	*/
    /**
     * approves a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun approvePost(postId: String): Completable {
        return PostApproveUseCase().execute(postId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: post.decline
	*/
    /**
     * declines a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun declinePost(postId: String): Completable {
        return PostDeclineUseCase().execute(postId)
    }
    /* end_public_function */

    @Deprecated("This function is deprecated, please use AmityPostRepository.flagPost(postId: String, reason: AmityContentFlagReason) instead",
        replaceWith = ReplaceWith(
            "flagPost(postId: String, reason: AmityContentFlagReason)",
            "com.amity.socialcloud.sdk.api.social.post.AmityPostRepository"
        ),
    )
    fun flagPost(postId: String): Completable {
        return flagPost(postId, AmityContentFlagReason.Others())
    }

    /* begin_public_function
      id: post.flag
    */
    /**
     * flags a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun flagPost(postId: String, reason: AmityContentFlagReason): Completable {
        val request = when(reason){
            is AmityContentFlagReason.Others -> {
                FlagContentRequest(
                    reason = reason.reason,
                    details = reason.details
                )
            }
            else -> {
                FlagContentRequest(
                    reason = reason.reason,
                )
            }
        }
        return FlagPostUseCase().execute(postId, request)
    }
    /* end_public_function */

    /* begin_public_function
      id: post.unflag
    */
    /**
     * unflags a post
     * @param [postId] a postId
     * @return Async completion as Completable
     */
    fun unflagPost(postId: String): Completable {
        return UnFlagPostUseCase().execute(postId)
    }
    /* end_public_function */

    fun getPinnedPosts(
        communityId: String,
        placement: String? = null,
        sortByOptions: AmityPinSortByOptions = AmityPinSortByOptions.lastPinned
    ): Flowable<PagingData<AmityPinnedPost>> {
        return PinnedPostQueryUseCase().execute(communityId, placement, sortByOptions)
    }

    fun getGlobalPinnedPosts(): Flowable<List<AmityPinnedPost>> {
        return GlobalPinnedPostQueryUseCase().execute()
    }

    fun semanticSearchPosts(
        query: String,
        targetType: AmityPost.TargetType?,
        targetId: String?,
        postTypes: List<AmityPost.DataType> = listOf(AmityPost.DataType.TEXT, AmityPost.DataType.IMAGE),
        matchingOnlyParentPosts: Boolean = true,
    ): Flowable<PagingData<AmityPost>> {
        return SemanticSearchPostUseCase().execute(
            query = query,
            targetType = targetType,
            targetId = targetId,
            matchingOnlyParentPost = matchingOnlyParentPosts,
            postTypes = postTypes
        )
    }

    /**
     * Initiates a function chain to create a post.
     * @param [postId] a postId
     * @return new instance of AmityPostCreationTarget.
     */
    @Deprecated("Use dedicated post creation function with dedicated post type instead")
    fun createPost(): AmityPostCreationTarget {
        return AmityPostCreationTarget()
    }

}
