package com.ekoapp.ekosdk.internal.usecase.comment

import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.amity.socialcloud.sdk.api.social.comment.query.AmityCommentSortOption
import com.amity.socialcloud.sdk.social.data.comment.CommentRepository
import com.amity.socialcloud.sdk.social.domain.comment.CommentComposerUseCase
import io.reactivex.rxjava3.core.Flowable

internal class CommentManualQueryUseCase {

    fun execute(
        referenceType: String,
        referenceId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        isDeleted: Boolean?,
        sortOption: AmityCommentSortOption,
        ids: List<String>,
        hasMore: Boolean
    ): Flowable<List<AmityComment>> {
        return getDataSource(
            referenceType,
            referenceId,
            isFilterByParentId,
            parentId,
            isDeleted,
            sortOption,
            ids,
            hasMore
        )
    }

    private fun getDataSource(
        referenceType: String,
        referenceId: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        isDeleted: Boolean?,
        sortOption: AmityCommentSortOption,
        ids: List<String>,
        hasMore: Boolean
    ): Flowable<List<AmityComment>> {
        lateinit var source: Flowable<List<AmityComment>>
        if (ids.isEmpty() && hasMore) {
            source = Flowable.never()
        } else if (sortOption == AmityCommentSortOption.LAST_CREATED) {
            source = getCombinedResults(
                referenceType,
                referenceId,
                isFilterByParentId,
                isDeleted,
                parentId,
                sortOption,
                ids
            )
        } else {
            if (hasMore) {
                source = observeCommentByIds(ids, isDeleted, sortOption)
            } else {
                source = getCombinedResults(
                    referenceType,
                    referenceId,
                    isFilterByParentId,
                    isDeleted,
                    parentId,
                    sortOption,
                    ids
                )
            }
        }

        return source

    }

    private fun getCombinedResults(
        referenceType: String,
        referenceId: String,
        isFilterByParentId: Boolean,
        isDeleted: Boolean?,
        parentId: String?,
        sortOption: AmityCommentSortOption,
        ids: List<String>
    ): Flowable<List<AmityComment>> {
        return Flowable.combineLatest(
            observeCommentByIds(
                ids,
                isDeleted,
                sortOption
            ).map { list ->
                list.filter { it.isDeleted() == isDeleted || isDeleted == null }
            },
            observeLocalComments(
                referenceType,
                referenceId,
                isFilterByParentId,
                isDeleted,
                parentId,
                sortOption,
                ids
            ).map { list ->
                list.filter { it.isDeleted() == isDeleted || isDeleted == null }
            }
        ) { commentByIds, localComments ->
            if (sortOption == AmityCommentSortOption.LAST_CREATED) {
                val comments = mutableListOf<AmityComment>()
                comments.addAll(localComments)
                comments.addAll(commentByIds)
                comments
            } else {
                // reverse adding order
                val comments = mutableListOf<AmityComment>()
                comments.addAll(commentByIds)
                comments.addAll(localComments)
                comments
            }
        }
    }

    private fun observeCommentByIds(
        ids: List<String>,
        isDeleted: Boolean?,
        sortOption: AmityCommentSortOption
    ): Flowable<List<AmityComment>> {
        return CommentRepository().getCommentCollection(
            ids,
            isDeleted,
            sortOption
        ).map { list ->
                list.map {
                    CommentComposerUseCase().execute(it)
                }
                list
            }
    }

    private fun observeLocalComments(
        referenceType: String,
        referenceId: String,
        isFilterByParentId: Boolean,
        isDeleted: Boolean?,
        parentId: String?,
        sortOption: AmityCommentSortOption,
        ids: List<String>
    ): Flowable<List<AmityComment>> {
        if (ids.isEmpty()) {
            return CommentRepository().observeCommentAfter(
                referenceType,
                referenceId,
                parentId,
                isFilterByParentId,
                isDeleted,
                sortOption,
                null
            )
                .map { list ->
                    list.map {
                        CommentComposerUseCase().execute(it)
                    }
                    list
                }
        } else {
            val offsetCommentId =
                if (sortOption == AmityCommentSortOption.LAST_CREATED) ids.first() else ids.last()
            return CommentRepository().observeCommentAfter(
                referenceType,
                referenceId,
                parentId,
                isFilterByParentId,
                isDeleted,
                sortOption,
                offsetCommentId
            )
                .map { list ->
                    list.map {
                        CommentComposerUseCase().execute(it)
                    }
                    list
                }
        }
    }

}