package com.amity.socialcloud.sdk.social.data.post.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.data.post.PostQueryPersister
import com.amity.socialcloud.sdk.social.data.post.PostRemoteDataStore
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.api.dto.EkoPostQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.usecase.post.PostReactorUseCase
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@ExperimentalPagingApi
internal class CommunityPostMediator(
    private val communityId: String,
    private val sortBy: String,
    private val isDeleted: Boolean?,
    private val feedType: AmityFeedType,
    private val dataTypes: List<AmityPost.DataType>,
    private val matchingOnlyParentPost: Boolean,
    private val includeMixedStructure: Boolean
) : DynamicQueryStreamMediator<PostEntity, EkoPostQueryDto, AmityPost>(
    nonce = AmityNonce.COMMUNITY_FEED,
    dynamicQueryStreamKeyCreator = CommunityPostKeyCreator(
        communityId = communityId,
        sortBy = sortBy,
        isDeleted = isDeleted,
        feedType = feedType,
        dataTypes = dataTypes,
        matchingOnlyParentPosts = matchingOnlyParentPost,
        includeMixedStructure = includeMixedStructure
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityPost> {
        return PostReactorUseCase().execute(
            targetType = AmityPost.TargetType.COMMUNITY,
            targetId = communityId,
            isDeleted = isDeleted,
            dataTypes = dataTypes,
            feedType = feedType,
            includeMixedStructure = includeMixedStructure,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = AmityNonce.COMMUNITY_FEED,
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EkoPostQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<EkoPostQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EkoPostQueryDto): Completable {
        return PostQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EkoPostQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.paging.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.posts?.map { it.postId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EkoPostQueryDto> {
        return PostRemoteDataStore().getCommunityPosts(
            targetId = communityId,
            isDeleted = isDeleted,
            feedType = feedType.apiKey,
            dataTypes = getDataTypes(),
            sortBy = sortBy,
            matchingOnlyParentPost = matchingOnlyParentPost,
            includeMixedStructure = includeMixedStructure,
            limit = limit,
            token = token
        ).toRx2()
    }

    private fun getDataTypes(): List<String>? {
        if (dataTypes.isNotEmpty()) {
            val typeArrayList = arrayListOf<String>()
            dataTypes.forEach { typeArrayList.add(it.getApiKey()) }
            return typeArrayList
        }
        return null
    }
}