package com.amity.socialcloud.sdk.infra.mqtt

import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.infra.mqtt.listener.MqttEventListeners
import com.amity.socialcloud.sdk.infra.mqtt.payload.MqttPayload
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.events.AmityTopic
import com.ekoapp.ekosdk.internal.data.converter.EkoGson
import com.ekoapp.ekosdk.internal.api.EkoEndpoint
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.hivemq.client.mqtt.MqttClient
import com.hivemq.client.mqtt.MqttGlobalPublishFilter
import com.hivemq.client.mqtt.datatypes.MqttQos
import com.hivemq.client.mqtt.mqtt3.Mqtt3RxClient
import com.hivemq.client.mqtt.mqtt3.exceptions.Mqtt3ConnAckException
import com.hivemq.client.mqtt.mqtt3.message.connect.Mqtt3Connect.DEFAULT_KEEP_ALIVE
import com.hivemq.client.mqtt.mqtt3.message.connect.connack.Mqtt3ConnAckReturnCode
import com.hivemq.client.mqtt.mqtt3.message.subscribe.Mqtt3Subscribe
import com.hivemq.client.mqtt.mqtt3.message.unsubscribe.Mqtt3Unsubscribe
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.disposables.CompositeDisposable
import io.reactivex.rxjava3.disposables.Disposable
import io.reactivex.rxjava3.processors.PublishProcessor
import io.reactivex.rxjava3.schedulers.Schedulers

internal class AmityMqttClient(
        sessionLifeCycleEventBus: SessionLifeCycleEventBus,
        sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
    
    private val connectRelay = PublishProcessor.create<EkoAccount>()
    
    init {
        connectRelay
            .concatMapCompletable { account ->
                activeClient?.let { client ->
                    obsoleteClient(client)
                            .onErrorComplete()
                            .andThen(connect(account))
                } ?: connect(account).onErrorComplete()
            }
            .subscribeOn(Schedulers.io())
            .doOnError {
                AmityLog.tag(TAG).e( "Fail to connect with error: $it")
            }
            .subscribe()
    }
    
    private val autoSubscribeTopics = listOf(
            AmityTopic.NETWORK(),
            AmityTopic.SMART_CHANNEL(),
            AmityTopic.SMART_MESSAGE_FEED(),
            AmityTopic.SMART_MESSAGE(),
            AmityTopic.USER_MARKER(),
            AmityTopic.LIVESTREAM(),
    )
    
    data class AuthenticatedMqttClient(
            val clientId: String,
            val account: EkoAccount,
            val mqttClient: Mqtt3RxClient
    )
    
    override fun onSessionStateChange(sessionState: SessionState) {
        AmityLog.tag("SSM3").e( "mqtt session change: $sessionState")
    }
    
    override fun establish(account: EkoAccount) {
        AmityLog.tag("SSM3").e( "mqtt session establish: ${this.hashCode()}")
        //always disconnect mqtt first, prevent the case that
        //user login with the same user
        connectRelay.onNext(account)
    }
    
    override fun destroy() {
        activeClient?.let {
            obsoleteClient(it)
                .subscribeOn(Schedulers.io())
                .subscribe()
        }
    }
    
    override fun handleTokenExpire() {
        activeClient?.let {
            obsoleteClient(it)
                .subscribeOn(Schedulers.io())
                .subscribe()
        }
    }
    
    fun disconnect(): Completable {
        return activeClient?.let(::obsoleteClient) ?: Completable.complete()
    }
    
    private fun connect(account: EkoAccount): Completable {
        val clientId = generateClientId(account)
        val userDao = UserDatabase.get().userDao()
        val user = userDao.getByIdNow(account.userId)
        val username = user?.mid ?: ""
        val password = account.accessToken.toByteArray()
        val mqttClient = try {
            initMqttClient(clientId, username, password)
        } catch(e: Exception) {
            // To avoid crash from Android 11 issue from this topic
            // https://issuetracker.google.com/issues/204913332
            // Which is inside HiveMQ. So, In order to handle the issue from SDK side.
            // When fail to initialize, the MQTT client will be null
            null
        }
        
        return mqttClient?.let {
            mqttClient.connectWith()
                    .cleanSession(false)
                    .keepAlive(DEFAULT_KEEP_ALIVE)
                    .applyConnect()
                    .ignoreElement()
                    .doOnSubscribe {
                        AmityLog.tag(TAG).e( "Connecting client: " + clientId + " userId: " + account.userId)
                    }
                    .doOnComplete {
                        AmityLog.tag(TAG).e( "Connected client: " + clientId + " userId: " + account.userId)
                        val authClient = AuthenticatedMqttClient(clientId, account, mqttClient)
                        activeClient = authClient
                        addClientListeners(authClient)
                        autoSubscribe()
                    }
                    .doOnError {
                        AmityLog.tag(TAG).e( "Connection exception: " + it.message)
                    }.toRx3()
        } ?: Completable.complete()
    }
    
    private fun generateClientId(newAccount: EkoAccount): String {
        return newAccount.deviceId
    }
    
    private fun initMqttClient(clientId: String, username: String, password: ByteArray): Mqtt3RxClient {
        return MqttClient.builder()
                .useMqttVersion3()
                .identifier(clientId)
                .serverHost(EkoEndpoint.getMqttUrl())
                .serverPort(443)
                .sslWithDefaultConfig()
                .simpleAuth()
                .username(username)
                .password(password)
                .applySimpleAuth()
                .automaticReconnectWithDefaultConfig()
                .addConnectedListener {
                    AmityLog.tag(TAG).e( "mqtt connected")
                }
                .addDisconnectedListener {
                    val exception = it.cause
                    val isNotActive = activeClient!= null && activeClient?.clientId != clientId
                    val hasValidDisconnectReason = exception is Mqtt3ConnAckException
                            &&
                            (exception.mqttMessage.returnCode == Mqtt3ConnAckReturnCode.IDENTIFIER_REJECTED
                                    || exception.mqttMessage.returnCode == Mqtt3ConnAckReturnCode.UNSUPPORTED_PROTOCOL_VERSION
                                    || exception.mqttMessage.returnCode == Mqtt3ConnAckReturnCode.BAD_USER_NAME_OR_PASSWORD
                                    || exception.mqttMessage.returnCode == Mqtt3ConnAckReturnCode.NOT_AUTHORIZED)
                    
                    val shouldNotReconnect = isNotActive || hasValidDisconnectReason
                    if (shouldNotReconnect) {
                        it.reconnector.reconnect(false)
                    }
                    
                    AmityLog.tag(TAG).e(
                            "mqtt disconnected || isNotActive: $isNotActive || hasValidDisconnectReason: $hasValidDisconnectReason "
                    )
                    AmityLog.tag(TAG).e("mqtt disconnected || cause: $exception")
                }
                .buildRx()
    }
    
    //obsolete will disconnect and terminate the current mqtt client
    private fun obsoleteClient(authClient: AuthenticatedMqttClient): Completable {
        subscriptions.clear()
        activeClient = null
        return  authClient.mqttClient.disconnect().toRx3()
                .subscribeOn(Schedulers.io())
                .doOnSubscribe {
                    AmityLog.tag(TAG).e(
                            "Disconnecting client: \" + ${authClient.clientId} + \" userId : \" + ${authClient.account.userId}"
                    )
                }
                .doOnComplete {
                    AmityLog.tag(TAG).e(
                            "Disconnected client: \" + ${authClient.clientId} + \" userId : \" + ${authClient.account.userId}"
                    )
                }
                .doOnError {
                    AmityLog.tag(TAG).e( "Disconnect error: " + it.message)
                }
    }
    
    private fun createEventSubscription(authClient: AuthenticatedMqttClient): Disposable {
        val mqttClient = authClient.mqttClient
        return mqttClient
                .publishes(MqttGlobalPublishFilter.ALL)
                .toRx3()
                .subscribeOn(Schedulers.io())
                .doOnNext {
                    val isNotActiveClient = authClient.clientId != activeClient?.clientId
                    if (isNotActiveClient) {
                        obsoleteClient(authClient)
                                .subscribeOn(Schedulers.io())
                                .subscribe()
                        return@doOnNext
                    }
                    try {
                        val payload = String(it.payloadAsBytes)
                        AmityLog.tag(TAG).d("received event: " + payload)
                        val event = EkoGson.get().fromJson(payload, MqttPayload::class.java)
                        val listener = MqttEventListeners.getMap()[event.eventType]
                        if (listener != null) {
                            event.data?.let { data ->
                                listener.onEvent(data)
                            }
                        } else {
                            //AmityLog.e(TAG, "Listener not found for event: " + event.eventType)
                        }
                    } catch (exception: Exception) {
                        //AmityLog.e(TAG, "Payload processing error: " + exception.message)
                    }
                    
                }
                .doOnError {
                    //AmityLog.e(TAG, "Unexpected Mqtt error")
                }
                .subscribe()
    }
    
    private fun addClientListeners(authClient: AuthenticatedMqttClient) {
        val eventSubscription = createEventSubscription(authClient)
        subscriptions.add(eventSubscription)
    }
    
    private fun autoSubscribe() {
        autoSubscribeTopics.forEach { topic ->
            val topicSubscription = subscribe(topic)
                    .subscribeOn(Schedulers.io())
                    .doOnError {
                        AmityLog.tag(TAG).e("Failed to subscribe " + topic.nonce)
                    }
                    .subscribe()
            subscriptions.add(topicSubscription)
        }
    }
    
    companion object {
        private val subscriptions = CompositeDisposable()
        private var activeClient: AuthenticatedMqttClient? = null
        val TAG = "AmityMqtt"
        
        fun subscribe(mqttTopic: AmityTopic): io.reactivex.rxjava3.core.Completable {
            return mqttTopic.generateTopic()
                    .flatMapCompletable { topic ->
                        getCurrentClient().flatMapCompletable { client ->
                            val subscribeMessage = Mqtt3Subscribe.builder()
                                    .topicFilter(topic)
                                    .qos(MqttQos.AT_LEAST_ONCE)
                                    .build()
                            client.subscribe(subscribeMessage)
                                    .toRx3()
                                    .doOnSuccess {
                                        AmityLog.tag(TAG).d("Subscribed to $topic")
                                    }
                                    .onErrorResumeNext {
                                        AmityLog.tag(TAG).e("Failed to subscribe $topic")
                                        val exception = AmityException.create(
                                                "Failed to subscribe",
                                                null,
                                                AmityError.UNKNOWN
                                        )
                                        Single.error(exception)
                                    }
                                    .ignoreElement()
                        }
                    }
            
        }
        
        fun unsubscribe(mqttTopic: AmityTopic): Completable {
            return mqttTopic.generateTopic()
                    .flatMapCompletable { topic ->
                        getCurrentClient()
                                .flatMapCompletable {
                                    val unsubscribeMessage = Mqtt3Unsubscribe.builder()
                                            .topicFilter(topic)
                                            .build()
                                    it.unsubscribe(unsubscribeMessage).toRx3()
                                }
                    }
        }
        
        private fun getCurrentClient(): Single<Mqtt3RxClient> {
            val client = activeClient?.mqttClient
            return if (client == null) {
                val exception =
                        AmityException.create(
                                "Failed to subscribe",
                                null,
                                AmityError.UNKNOWN
                        )
                Single.error(exception)
            } else {
                Single.just(client)
            }
        }
    }
}