package com.amity.socialcloud.sdk.social.data.event

import androidx.paging.Pager
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import androidx.paging.map
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.entity.social.event.EventEntity
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.event.AmityEvent
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventRSVPStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.amity.socialcloud.sdk.social.data.event.paging.EventQueryPagingSource
import com.amity.socialcloud.sdk.social.data.event.paging.MyEventsPagingSource
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import kotlinx.coroutines.rx3.asFlowable
import org.joda.time.DateTime

internal class EventRepository : AmityObjectRepository<EventEntity, AmityEvent>() {

    override fun fetchAndSave(objectId: String): Completable {
        return EventRemoteDataStore().getEvent(objectId)
            .flatMapCompletable {
                EventQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): EventEntity? {
        return EventLocalDataStore().getEvent(objectId)
    }

    override fun mapper(): ModelMapper<EventEntity, AmityEvent> {
        return EventModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EventEntity> {
        return EventLocalDataStore().observeEvent(objectId)
    }

    fun getEventPagingData(
        originType: AmityEventOriginType?,
        originId: String?,
        userId: String?,
        status: AmityEventStatus?,
        type: AmityEventType?,
        onlyAttended: Boolean?
    ): Flowable<PagingData<AmityEvent>> {
        val pager = Pager(
            config = PagingConfig(
                pageSize = 20,
                enablePlaceholders = false,
                initialLoadSize = 20
            ),
            pagingSourceFactory = {
                EventQueryPagingSource(
                    originType = originType,
                    originId = originId,
                    userId = userId,
                    status = status,
                    type = type,
                    onlyAttended = onlyAttended
                )
            }
        )
        
        return pager.flow.asFlowable()
            .map { pagingData ->
                pagingData.map { entity ->
                    EventModelMapper().map(entity)
                }
            }
    }

    fun getMyEventsPagingData(
        status: AmityEventRSVPStatus?
    ): Flowable<PagingData<AmityEvent>> {
        val pager = Pager(
            config = PagingConfig(
                pageSize = 20,
                enablePlaceholders = false,
                initialLoadSize = 20
            ),
            pagingSourceFactory = {
                MyEventsPagingSource(
                    status = status
                )
            }
        )
        
        return pager.flow.asFlowable()
            .map { pagingData ->
                pagingData.map { entity ->
                    EventModelMapper().map(entity)
                }
            }
    }

    fun createEvent(
        title: String?,
        description: String?,
        type: AmityEventType?,
        isInviteOnly: Boolean?,
        startTime: DateTime?,
        endTime: DateTime, // Made non-nullable (required)
        originType: AmityEventOriginType,
        originId: String,
        location: String?,
        externalUrl: String?,
        coverImageFileId: String?,
        tags: AmityTags?,
        timezone: String?
    ): Single<AmityEvent> {
        return EventRemoteDataStore().createEvent(
            title = title,
            description = description,
            type = type,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            originType = originType,
            originId = originId,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags,
            timezone = timezone
        ).flatMap { dto ->
            EventQueryPersister().persist(dto)
                .andThen(
                    dto.events?.firstOrNull()?.eventId?.let { eventId ->
                        // For now, directly map from DTO to model as we don't have local caching
                        val entity = dto.events?.firstOrNull()?.let { EventEntityMapper().map(listOf(it)).firstOrNull() }
                        entity?.let { Single.just(EventModelMapper().map(it)) }
                            ?: Single.error(Exception("Event not found in response"))
                    } ?: Single.error(Exception("No events in response"))
                )
        }
    }

    fun updateEvent(
        eventId: String,
        title: String?,
        description: String?,
        isInviteOnly: Boolean?,
        startTime: DateTime?,
        endTime: DateTime?,
        location: String?,
        externalUrl: String?,
        coverImageFileId: String?,
        tags: AmityTags?,
        timezone: String?
    ): Single<AmityEvent> {
        return EventRemoteDataStore().updateEvent(
            eventId = eventId,
            title = title,
            description = description,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags,
            timezone = timezone
        ).flatMap { dto ->
            EventQueryPersister().persist(dto)
                .andThen(
                    dto.events?.firstOrNull()?.eventId?.let { eventId ->
                        // For now, directly map from DTO to model as we don't have local caching
                        val entity = dto.events?.firstOrNull()?.let { EventEntityMapper().map(listOf(it)).firstOrNull() }
                        entity?.let { Single.just(EventModelMapper().map(it)) }
                            ?: Single.error(Exception("Event not found in response"))
                    } ?: Single.error(Exception("No events in response"))
                )
        }
    }

    fun deleteEvent(eventId: String): Completable {
        return EventRemoteDataStore().deleteEvent(eventId)
            .flatMapCompletable {
                when (it.isSuccess) {
                    true -> {
                        EventLocalDataStore().deleteEvent(eventId)
                    }
                    else -> Completable.complete()
                }
            }
    }
}
