package com.amity.socialcloud.sdk.chat.domain.message

import com.amity.socialcloud.sdk.chat.data.marker.message.MessageMarkerRepository
import com.amity.socialcloud.sdk.core.domain.ComposerUseCase
import com.amity.socialcloud.sdk.core.domain.file.AudioGetUseCase
import com.amity.socialcloud.sdk.core.domain.file.FileGetUseCase
import com.amity.socialcloud.sdk.core.domain.file.ImageGetNowUseCase
import com.amity.socialcloud.sdk.core.domain.file.ImageGetUseCase
import com.amity.socialcloud.sdk.core.domain.file.VideoGetUseCase
import com.amity.socialcloud.sdk.core.domain.reaction.MyReactionsGetUseCase
import com.amity.socialcloud.sdk.core.domain.user.UserGetNowUseCase
import com.amity.socialcloud.sdk.core.domain.user.UserGetUseCase
import com.amity.socialcloud.sdk.helper.core.mention.AmityMentionee
import com.amity.socialcloud.sdk.model.chat.message.AmityMessage
import com.amity.socialcloud.sdk.model.core.file.AmityFileInfo
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.ekoapp.ekosdk.internal.usecase.message.IsMessageFlaggedByMeNowUseCase
import com.ekoapp.ekosdk.internal.usecase.message.IsMessageFlaggedByMeUseCase
import com.google.gson.JsonObject


internal class MessageComposerUseCase : ComposerUseCase() {

    fun execute(message: AmityMessage): AmityMessage {
        addMyReactions(message)
        addMyFlag(message)
        addUser(message)
        addMentioneeUser(message)
        addData(message)
        addMarker(message)
        return message
    }

    private fun addMyReactions(message: AmityMessage) {
        message.myReactions =
            MyReactionsGetUseCase().execute(AmityReactionReferenceType.MESSAGE, message.getMessageId())
    }

    private fun addMyFlag(message: AmityMessage) {
        message.isFlaggedByMe = IsMessageFlaggedByMeNowUseCase().execute(message.getMessageId())
    }

    private fun addUser(message: AmityMessage) {
        message.user = UserGetNowUseCase().execute(message.getCreatorId())
    }

    private fun addMentioneeUser(message: AmityMessage) {
        message.mentionees = message.getMentionees().map {
            if (it is AmityMentionee.USER) {
                it.user = UserGetNowUseCase().execute(it.getUserId())
            }
            it
        }
    }

    private fun addData(message: AmityMessage) {
        val file = getFile(message)
        val thumbnailFile = getThumbnailImage(message)
        val data = AmityMessage.Data.from(
            message.getMessageId(),
            message.getDataType(),
            message.rawData ?: JsonObject(),
            file,
            thumbnailFile
        )
        message.data = data
    }

    private fun getFile(message: AmityMessage): AmityFileInfo? {
        val fileId = message.rawData?.get("fileId")?.asString
        return fileId?.let {
            when(message.getDataType()) {
                AmityMessage.DataType.IMAGE -> {
                ImageGetNowUseCase().execute(it)
            }
                AmityMessage.DataType.FILE-> {
                FileGetUseCase().execute(it)
            }
                AmityMessage.DataType.AUDIO -> {
                AudioGetUseCase().execute(it)
            }
                AmityMessage.DataType.VIDEO -> {
                VideoGetUseCase().execute(it)
            }
                else -> null
            }
        }
    }

    private fun getThumbnailImage(message: AmityMessage): AmityImage? {
        val thumbnailFileId = message.rawData?.get("thumbnailFileId")?.asString
        return thumbnailFileId?.let {
            ImageGetNowUseCase().execute(it)
        }
    }
    
    private fun addMarker(message: AmityMessage) {
        message.readCount = MessageMarkerRepository().getReadCount(message.getMessageId())
        message.deliveredCount = MessageMarkerRepository().getDeliveredCount(message.getMessageId())
    }

}