package com.ekoapp.ekosdk.internal.usecase

import com.amity.socialcloud.sdk.core.data.tombstone.TombstoneRepository
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.ekoapp.ekosdk.EkoObject
import com.ekoapp.ekosdk.internal.TombstoneModelType
import com.ekoapp.ekosdk.internal.entity.TombstoneEntity
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

internal abstract class LiveObjectUseCase<Entity : EkoObject, PublicModel : Any> {

    private val nullTombstone = TombstoneEntity().apply {
        this.objectId = "NULL_TOMBSTONE"
    }

    abstract fun createRepository(): AmityObjectRepository<Entity, PublicModel>

    //return null if the object doesn't need to be composed
    abstract fun composeModel(model: PublicModel): PublicModel?

    abstract fun tombstoneModelType(): TombstoneModelType

    open fun execute(objectId: String): Flowable<PublicModel> {
        return verifyObjectTombstone(objectId)
            .flatMapPublisher { isObjectValid ->
                if (isObjectValid) {
                    return@flatMapPublisher retrieveLiveObject(objectId)
                } else {
                    // if the object is not valid
                    // meaning that the object was hard deleted
                    // immediately throw AmityError.ITEM_NOT_FOUND exception
                    throw
                    AmityException.create(
                        message = "item not found",
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                }
            }
                .subscribeOn(Schedulers.io())
    }

    private fun retrieveLiveObject(objectId: String): Flowable<PublicModel> {
        val obtainObject = createRepository().obtain(objectId)
            .ignoreElement()
            .doOnError {
                // if the server returns AmityError.ITEM_NOT_FOUND
                // cache in tombstone temporarily (depends on expiration duration)
                if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                    TombstoneRepository().saveTombstone(
                        objectId = objectId,
                        tombstoneModelType = tombstoneModelType(),
                        amityError = AmityError.ITEM_NOT_FOUND
                    ).subscribeOn(Schedulers.io())
                        .subscribe()
                }
            }

        val observe = createRepository().observe(objectId)

        return obtainObject.andThen(observe)
            .map { result ->
                val composedModel = composeModel(result)
                if (composedModel != null) {
                    return@map composedModel
                } else {
                    return@map result
                }
            }
    }

    private fun verifyObjectTombstone(objectId: String): Single<Boolean> {
        return Single.fromCallable {
            TombstoneRepository().getTombstone(
                objectId = objectId,
                tombstoneModelType = tombstoneModelType()
            ) ?: nullTombstone
        }.map { tombstone ->
            val isExpired = tombstone.expiresAt?.isBeforeNow == true
            val isObjectValid = if (tombstone.objectId == nullTombstone.objectId) {
                // object doesn't exist in tombstone
                // meaning that the object is valid and never hard deleted
                true
            } else {
                // object exist in tombstone, we need to double check
                // if the tombstone is expired or not.
                // if tombstone is not expired, immediately return 400400 error
                isExpired
            }
            return@map isObjectValid
        }
    }
}

