package com.ekoapp.ekosdk.internal.push

import android.util.Pair
import com.amity.socialcloud.sdk.log.AmityLog.e
import com.amity.socialcloud.sdk.log.AmityLog.tag
import com.amity.socialcloud.sdk.push.EkoPushContract
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.EkoNotificationApi
import com.ekoapp.ekosdk.internal.api.http.request.RegisterDeviceForPushNotificationRequest
import com.ekoapp.ekosdk.internal.api.http.request.RegisterDeviceForPushNotificationRequest.BaiduToken
import com.ekoapp.ekosdk.internal.api.http.request.RegisterDeviceForPushNotificationResponse
import com.ekoapp.ekosdk.internal.api.http.request.UnregisterDeviceForPushNotificationRequest
import com.ekoapp.ekosdk.internal.api.http.request.UnregisterDeviceForPushNotificationResponse
import com.ekoapp.ekosdk.internal.data.EkoDatabase
import com.ekoapp.ekosdk.internal.data.model.*
import com.github.davidmoten.guavamini.Objects
import com.google.gson.Gson
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers

class EkoPushContractImpl : EkoPushContract {
	private val TAG = javaClass.name
	
	internal enum class Provider(val apiKey: String) {
		FCM("fcm"), BAIDU("baidu");
		
	}
	
	private fun initPushService() {
		Flowable.combineLatest(
			initFcm().startWithItem(TokenConfigAndApiKey(null, null, null)),
			initBaidu().startWithItem(TokenConfigAndApiKey(null, null, null))
		) { fcmConfig, baiduConfig -> Pair(fcmConfig, baiduConfig) }.skip(1)
			.flatMapCompletable { configPair: Pair<TokenConfigAndApiKey<EkoFcmToken?>, TokenConfigAndApiKey<EkoBaiduToken?>> ->
				val fcmConfig = configPair.first
				var fcmState = fcmConfig.config?.state?.apiKey ?: "null"
				var fcmToken = fcmConfig.token?.token ?: "null"
				e("fcm: $fcmState token:$fcmToken")
				val baiduConfig = configPair.second
				var baiduState = baiduConfig.config?.state?.apiKey ?: "null"
				var baiduToken =  baiduConfig.token?.token ?: "null"
				e("baidu: $baiduState token:$baiduToken")
				if (fcmConfig.token != null && baiduConfig.token != null) {
					return@flatMapCompletable prioritizeFcmToken(fcmConfig, baiduConfig)
				} else if (fcmConfig.token != null) {
					return@flatMapCompletable handleFcmToken(fcmConfig)
				} else {
					return@flatMapCompletable handleBaiduToken(baiduConfig)
				}
			}
			.subscribeOn(Schedulers.io())
			.onErrorComplete()
			.subscribe()
	}
	
	/**
	 * comment
	 */
	init {
		initPushService()
	}
	
	private fun initFcm(): Flowable<TokenConfigAndApiKey<EkoFcmToken?>> {
		return Flowable.combineLatest(
			EkoDatabase.get().fcmTokenDao().fcmToken.distinctUntilChanged { oldToken, newToken -> oldToken == newToken },
			getPushConfig(),
			getApiKey()
		) { token: EkoFcmToken, config: EkoPushConfig?, apiKey: EkoApiKey? ->
			TokenConfigAndApiKey(token, config, apiKey)
		}
	}
	
	private fun initBaidu(): Flowable<TokenConfigAndApiKey<EkoBaiduToken?>> {
		return Flowable.combineLatest<EkoBaiduToken, EkoPushConfig, EkoApiKey, TokenConfigAndApiKey<EkoBaiduToken?>>(
			EkoDatabase.get().baiduTokenDao().baiduToken.distinctUntilChanged { oldToken, newToken -> oldToken == newToken },
			getPushConfig(),
			getApiKey()
		) { token: EkoBaiduToken?, config: EkoPushConfig?, apiKey: EkoApiKey? ->
			TokenConfigAndApiKey(token, config, apiKey)
		}
		.filter { tokenConfigAndApiKey: TokenConfigAndApiKey<EkoBaiduToken?> -> tokenConfigAndApiKey.token?.userId != null && tokenConfigAndApiKey.token?.channelId != null }
	}
	
	private fun prioritizeFcmToken(fcmConfig: TokenConfigAndApiKey<EkoFcmToken?>, baiduConfig: TokenConfigAndApiKey<EkoBaiduToken?>): Completable {
		return if (Objects.equal(EkoPushConfig.State.REGISTERED, fcmConfig.config?.state)) {
			val registerRequest = RegisterDeviceForPushNotificationRequest.create(fcmConfig.config?.userId,
				fcmConfig.config?.deviceId,
				fcmConfig.token?.token!!,
				Provider.FCM.apiKey)
			val unregisterRequest = UnregisterDeviceForPushNotificationRequest.create(baiduConfig.config?.userId,
				baiduConfig.config?.deviceId)
			AmityHttpClient.get(EkoNotificationApi::class)
				.flatMap {
					it.unregisterNotificationToken(baiduConfig.apiKey!!.apiKey, unregisterRequest)
				}
				.ignoreElement()
				.andThen(
					AmityHttpClient.get(EkoNotificationApi::class)
						.flatMap {
							it.registerNotificationToken(fcmConfig.apiKey!!.apiKey, registerRequest)
						}
						.doOnSuccess { response: RegisterDeviceForPushNotificationResponse -> tag(TAG).i(String.format("fcm response:%s", Gson().toJson(response))) }
						.ignoreElement()
						.onErrorComplete()
				)
				.onErrorComplete()
		} else {
			val request = UnregisterDeviceForPushNotificationRequest.create(fcmConfig.config!!.userId,
					fcmConfig.config!!.deviceId)
			tag(TAG).i(String.format("fcm un-registration request:%s", Gson().toJson(request)))
			AmityHttpClient.get(EkoNotificationApi::class)
					.flatMap {
						it.unregisterNotificationToken(fcmConfig.apiKey!!.apiKey, request)
					}
					.doOnSuccess { response: UnregisterDeviceForPushNotificationResponse -> tag(TAG).i(String.format("fcm response:%s", Gson().toJson(response))) }
					.ignoreElement()
					.onErrorComplete()
		}
	}
	
	private fun handleFcmToken(tokenConfigAndApiKey: TokenConfigAndApiKey<EkoFcmToken?>): Completable {
		return if (Objects.equal(EkoPushConfig.State.REGISTERED, tokenConfigAndApiKey.config!!.state)) {
			val request = RegisterDeviceForPushNotificationRequest.create(tokenConfigAndApiKey.config!!.userId,
					tokenConfigAndApiKey.config!!.deviceId,
					tokenConfigAndApiKey.token!!.token,
					Provider.FCM.apiKey)
			tag(TAG).i(String.format("fcm registration request:%s", Gson().toJson(request)))
			AmityHttpClient.get(EkoNotificationApi::class)
					.flatMap {
						it.registerNotificationToken(tokenConfigAndApiKey.apiKey!!.apiKey, request)
					}
					.doOnSuccess { response: RegisterDeviceForPushNotificationResponse -> tag(TAG).i(String.format("fcm response:%s", Gson().toJson(response))) }
					.ignoreElement()
					.onErrorComplete()
		} else {
			val userId = tokenConfigAndApiKey.config!!.userId
			val request = UnregisterDeviceForPushNotificationRequest.create(tokenConfigAndApiKey.config!!.userId,
					tokenConfigAndApiKey.config!!.deviceId)
			tag(TAG).i(String.format("fcm un-registration request:%s", Gson().toJson(request)))
			AmityHttpClient.get(EkoNotificationApi::class)
					.flatMap {
						it.unregisterNotificationToken(tokenConfigAndApiKey.apiKey!!.apiKey, request)
					}
					.doOnSuccess { response: UnregisterDeviceForPushNotificationResponse -> tag(TAG).i(String.format("fcm response:%s", Gson().toJson(response))) }
					.ignoreElement()
					.andThen(Completable.fromAction { EkoDatabase.get().pushConfigDao().clearUnregisteredPushConfigForUser(userId) }) // Remove unregister success config from DB
					.onErrorComplete()
		}
	}
	
	private fun handleBaiduToken(tokenConfigAndApiKey: TokenConfigAndApiKey<EkoBaiduToken?>): Completable {
		return if (Objects.equal(EkoPushConfig.State.REGISTERED, tokenConfigAndApiKey.config!!.state)) {
			val request = RegisterDeviceForPushNotificationRequest.create(tokenConfigAndApiKey.config!!.userId,
					tokenConfigAndApiKey.config!!.deviceId,
					Gson().toJson(BaiduToken(tokenConfigAndApiKey.token!!.token, tokenConfigAndApiKey.token!!.userId, tokenConfigAndApiKey.token!!.channelId)),
					Provider.BAIDU.apiKey)
			tag(TAG).i(String.format("baidu registration request:%s", Gson().toJson(request)))
			AmityHttpClient.get(EkoNotificationApi::class)
					.flatMap {
						it.registerNotificationToken(tokenConfigAndApiKey.apiKey!!.apiKey, request)
					}
					.doOnSuccess { response: RegisterDeviceForPushNotificationResponse -> tag(TAG).i(String.format("baidu response:%s", Gson().toJson(response))) }
					.ignoreElement()
					.onErrorComplete()
		} else {
			val request = UnregisterDeviceForPushNotificationRequest.create(tokenConfigAndApiKey.config!!.userId,
					tokenConfigAndApiKey.config!!.deviceId)
			tag(TAG).i(String.format("baidu un-registration request:%s", Gson().toJson(request)))
			AmityHttpClient.get(EkoNotificationApi::class)
					.flatMap {
						it.unregisterNotificationToken(tokenConfigAndApiKey.apiKey!!.apiKey, request)
					}
					.doOnSuccess { response: UnregisterDeviceForPushNotificationResponse -> tag(TAG).i(String.format("baidu response:%s", Gson().toJson(response))) }
					.ignoreElement()
					.onErrorComplete()
		}
	}
	
	override fun insertBaiduApiKey(apiKey: String): Completable {
		return Completable.fromAction {
			EkoDatabase.get()
					.baiduTokenDao()
					.insertOrUpdate(EkoBaiduToken(apiKey))
		}
	}
	
	override fun updateBaiduToken(token: String?, userId: String?, channelId: String?): Completable {
		return Completable.fromAction {
			EkoDatabase.get()
					.baiduTokenDao()
					.updateToken(token, userId, channelId)
		}
	}
	
	override fun insertFcmToken(fcmToken: String): Completable {
		return Completable.fromAction { EkoDatabase.get().fcmTokenDao().insert(EkoFcmToken(fcmToken)) }
	}
	
	override fun hasRegisteredConfig(): Flowable<Boolean> {
		return EkoDatabase.get()
				.pushConfigDao()
				.hasRegisteredConfig()
	}
	
	private fun getPushConfig(): Flowable<EkoPushConfig> {
		return EkoDatabase.get()
				.accountDao()
				.all
				.flatMapIterable { accounts -> accounts }
				.distinct(EkoAccount::getUserId)
				.flatMap { account ->
					EkoDatabase.get()
							.pushConfigDao()
							.getPushConfig(account.userId)
							.distinctUntilChanged { config: EkoPushConfig -> config }
				}
	}
	
	internal inner class TokenConfigAndApiKey<TOKEN>(var token: TOKEN, var config: EkoPushConfig?, var apiKey: EkoApiKey?)
	companion object {
		private fun getApiKey(): Flowable<EkoApiKey> {
			return EkoDatabase.get()
				.apiKeyDao()
				.currentApiKeyFlowable
		}
	}
}