package com.amity.socialcloud.sdk.social.domain.comment

import com.amity.socialcloud.sdk.core.domain.ComposerUseCase
import com.amity.socialcloud.sdk.core.domain.file.ImageGetUseCase
import com.amity.socialcloud.sdk.core.domain.reaction.MyReactionsGetUseCase
import com.amity.socialcloud.sdk.core.domain.user.UserGetUseCase
import com.amity.socialcloud.sdk.helper.core.mention.AmityMentionee
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.amity.socialcloud.sdk.social.domain.comment.flag.IsCommentFlaggedByMeUseCase
import com.amity.socialcloud.sdk.social.domain.community.member.CommunityMemberGetUseCase
import com.google.gson.JsonObject

internal class CommentComposerUseCase : ComposerUseCase() {

    fun execute(comment: AmityComment): AmityComment {
        addLatestReplies(comment)
        addMyReactions(comment)
        addMyFlag(comment)
        addCommentUser(comment)
        addMentioneeUser(comment)
        addData(comment)
        addTarget(comment)
        return comment
    }

    private fun addMyReactions(comment: AmityComment) {
        comment.myReactions =
            MyReactionsGetUseCase().execute(
                AmityReactionReferenceType.COMMENT,
                comment.getCommentId()
            )
    }

    private fun addMyFlag(comment: AmityComment) {
        comment.isFlaggedByMe = IsCommentFlaggedByMeUseCase().execute(comment.getCommentId())
    }

    private fun addCommentUser(comment: AmityComment) {
        comment.user = UserGetUseCase().execute(comment.getCreatorId())
    }

    private fun addMentioneeUser(comment: AmityComment) {
        comment.mentionees = comment.getMentionees().map {
            if (it is AmityMentionee.USER) {
                it.user = UserGetUseCase().execute(it.getUserId())
            }
            it
        }
    }

    private fun addLatestReplies(comment: AmityComment) {
        comment.latestReplies = LatestRepliesGetUseCase().execute(
            referenceType = comment.referenceType,
            referenceId = comment.referenceId,
            parentId = comment.getCommentId(),
        )
    }

    private fun addData(comment: AmityComment) {
        comment.attachments = comment.rawAttachments.asJsonArray.mapNotNull {
            it.asJsonObject
        }.mapNotNull {
            getAttachment(it)
        }
    }

    private fun getAttachment(attachment: JsonObject): AmityComment.Attachment? {
        when (attachment.get("type").asString) {
            AmityComment.DataType.IMAGE.apiKey -> {
                val fileId = attachment.get("fileId")?.asString
                return fileId?.let {
                    AmityComment.Attachment.IMAGE(
                        fileId = fileId,
                        image = ImageGetUseCase().execute(it)
                    )
                }
            }
        }
        return null
    }

    private fun addTarget(comment: AmityComment) {
        when (val target = comment.target) {
            is AmityComment.Target.COMMUNITY -> {
                target.creatorCommunityMember = CommunityMemberGetUseCase().execute(
                    target.targetCommunityId,
                    target.creatorCommunityMemberId
                )
            }

            else -> {}
        }
    }
}