package com.amity.socialcloud.sdk.social.data.community

import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityMembershipStatusFilter
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityStorySettings
import com.amity.socialcloud.sdk.social.data.post.PostApi
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.api.dto.EkoDeletionDto
import com.ekoapp.ekosdk.internal.api.dto.EkoPostQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityCreateRequest
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityUpdateRequest
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single

internal class CommunityRemoteDataStore {

    fun createCommunity(
        displayName: String,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        userIds: List<String>?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?,
        storySettings: AmityCommunityStorySettings?
    ): Single<CommunityListQueryDto> {
        val request = CommunityCreateRequest(
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            userIds = userIds,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost,
            allowCommentInStory = storySettings?.allowComment
        )

        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.createCommunity(request)
        }
    }

    fun updateCommunity(
        communityId: String,
        displayName: String?,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?,
        storySettings: AmityCommunityStorySettings?
    ): Single<CommunityListQueryDto> {
        val request = CommunityUpdateRequest(
            communityId = communityId,
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost,
            allowCommentInStory = storySettings?.allowComment
        )
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.updateCommunity(communityId, request)
        }
    }

    fun joinCommunity(communityId: String): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.joinCommunity(communityId)
        }
    }

    fun leaveCommunity(communityId: String): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.leaveCommunity(communityId)
        }
    }

    fun deleteCommunity(communityId: String): Single<EkoDeletionDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.deleteCommunity(communityId)
        }
    }

    fun getCommunity(communityId: String): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getCommunity(communityId)
        }
    }

    fun getRecommendedCommunities(options: QueryOptionsRequestParams): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getRecommendedCommunities(
                limit = options.limit,
                token = options.token
            )
        }
    }

    fun getTrendingCommunities(options: QueryOptionsRequestParams): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getTrendingCommunities(
                limit = options.limit,
                token = options.token
            )
        }
    }

    fun queryCommunities(
        keyword: String? = null,
        categoryId: String? = null,
        filter: String? = null,
        sortBy: String? = null,
        isDeleted: Boolean? = null,
        options: QueryOptionsRequestParams
    ): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.queryCommunities(
                keyword = keyword,
                categoryId = categoryId,
                filter = filter,
                sortBy = sortBy,
                isDeleted = isDeleted,
                limit = options.limit,
                token = options.token
            )
        }
    }

    fun searchCommunities(
        query: String,
        filter: AmityCommunityMembershipStatusFilter?,
        tags: List<String>? = null,
        categoryIds: List<String>? = null,
        limit: Int? = null,
        token: String? = null
    ): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.searchCommunities(
                query = query,
                filter = filter?.value,
                tags = tags?.filter { it.isNotBlank() }.takeIf { it?.isNotEmpty() == true },
                categoryIds = categoryIds?.filter { it.isNotBlank() }.takeIf { it?.isNotEmpty() == true },
                limit = limit,
                token = token
            )
        }
    }

}