package com.amity.socialcloud.sdk.infra.upload

import android.media.MediaPlayer
import android.net.Uri
import co.amity.rxupload.extension.upload
import com.amity.socialcloud.sdk.core.data.file.RawFileModelMapper
import com.amity.socialcloud.sdk.model.core.content.AmityContentFeedType
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.file.AmityFileAccessType
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadInfo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.util.AppContext
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.Duration

private const val UPLOAD_VIDEO_PATH = "api/v4/videos"
private const val MULTIPART_DATA_KEY = "files"
private val MAX_DURATION = Duration.standardHours(2).millis // 2 Hours in milliseconds
private const val MAX_SIZE = 1000000000.0 // 1 Gigabyte

class AmityVideoUploadService private constructor(
    val uri: Uri,
    val uploadId: String?,
    val feedType: AmityContentFeedType? = null
) : AmityUploadService<AmityUploadResult<AmityVideo>>() {


    override fun getUploadParams(): Map<String, Any> {
        val params = mutableMapOf<String, Any>()
        params[AmityFileAccessType.apiKey] = getAccessType().apiKey
        if (feedType != null) {
            params[AmityContentFeedType.apiKey] = feedType.apiKey
        }
        return params
    }

    override fun getUploadHeaders(): Map<String, Any> {
        return mapOf()
    }

    override fun makeUploadServiceRequest(): Flowable<AmityUploadResult<AmityVideo>> {
        val contentResolver = AppContext.get().contentResolver
        
        // If android.media.MediaPlayer doesn't support file format
        // then let BE validate video duration instead
        val isValidDurationTime = try {
            var durationTime: Long
            MediaPlayer.create(AppContext.get(), uri).also { player ->
                durationTime = player.duration.toLong()
                player.reset()
                player.release()
            }
            durationTime <= MAX_DURATION
        } catch (e: Exception) {
            true
        }

        val fileDescriptor = contentResolver.openAssetFileDescriptor(uri, "r")
        val fileSize = fileDescriptor?.length

        return if (fileSize != null && fileSize <= MAX_SIZE) {
            uri.upload(
                context = AppContext.get(),
                path = UPLOAD_VIDEO_PATH,
                params = getUploadParams(),
                headers = getUploadHeaders(),
                id = uploadId,
                multipartDataKey = MULTIPART_DATA_KEY,
                accessType = getAccessType().apiKey,
            )
                .flatMap { fileProps ->
                    if (fileProps.progress == 100) {
                        Single.fromCallable {
                            val fileEntity = parseEkoFileEntity(fileProps.responseBody)
                            fileEntity.filePath = fileProps.uri.path
                            val fileDao = UserDatabase.get().fileDao()
                            fileDao.insert(fileEntity)
                            RawFileModelMapper().map(fileEntity)
                        }
                            .subscribeOn(Schedulers.io())
                            .flatMapPublisher {
                                Flowable.just(
                                    AmityUploadResult.COMPLETE(
                                        AmityVideo(it)
                                    )
                                )
                            }
                    } else {
                        Flowable.just(AmityUploadResult.PROGRESS(AmityUploadInfo(fileProps)))
                    }
                }
                .onErrorReturn {
                    val exception = parseErrorResponse(it)
                    AmityUploadResult.ERROR(exception)
                }
        } else {
            val exception =
                AmityException.create("File size exceeded!", null, AmityError.FILE_SIZE_EXCEEDED)
            Flowable.just(AmityUploadResult.ERROR(exception))
        }
    }

    class Builder {

        private lateinit var uri: Uri
        private var uploadId: String? = null
        private var feedType: AmityContentFeedType? = null

        internal fun fileUri(uri: Uri): Builder {
            this.uri = uri
            return this
        }

        fun uploadId(uploadId: String): Builder {
            this.uploadId = uploadId
            return this
        }

        internal fun feedType(feedType: AmityContentFeedType): Builder {
            this.feedType = feedType
            return this
        }

        fun build(): AmityVideoUploadService {
            return AmityVideoUploadService(uri, uploadId, feedType)
        }
    }
}