package com.ekoapp.ekosdk.internal.data.dao;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Query;

import com.amity.socialcloud.sdk.api.social.comment.query.AmityCommentSortOption;
import com.ekoapp.ekosdk.internal.PostEntity;
import com.ekoapp.ekosdk.internal.entity.CommentEntity;

import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Completable;
import io.reactivex.rxjava3.core.Flowable;

@Dao
public abstract class EkoCommentDao extends EkoObjectDao<CommentEntity> {

    @Query("SELECT comment.*," +
            " comment_flag.commentId as flag_commentId," +
            " comment_flag.flag as flag_flag," +
            " comment_flag.localFlag as flag_localFlag" +
            " from comment, comment_flag" +
            " where comment.referenceType = :referenceType " +
            " and comment.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " and comment.referenceId = :referenceId" +
            " and comment.commentId = comment_flag.commentId" +
            " order by case when :isSortByCreatedACS = 1 then comment.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then comment.createdAt end desc")
    abstract DataSource.Factory<Integer, CommentEntity> getAllImpl(String referenceType,
                                                                   String referenceId,
                                                                   Boolean isDeleted,
                                                                   Boolean isSortByCreatedACS);

    public DataSource.Factory<Integer, CommentEntity> getAll(String referenceType,
                                                             String referenceId,
                                                             Boolean isDeleted,
                                                             AmityCommentSortOption sortOption) {
        return getAllImpl(referenceType, referenceId, isDeleted, isSortByCreatedACS(sortOption));
    }

    @Query("SELECT comment.*" +
            " from comment" +
            " where comment.commentId in (:commentIds) " +
            " order by case when :isSortByCreatedACS = 1 then comment.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then comment.createdAt end desc")
    abstract Flowable<List<CommentEntity>> getAllByIdsImpl(List<String> commentIds, Boolean isSortByCreatedACS);

    public Flowable<List<CommentEntity>> getAllByIds(List<String> commentIds, AmityCommentSortOption sortOption) {
        return getAllByIdsImpl(commentIds, isSortByCreatedACS(sortOption));
    }

    @Query("SELECT comment.*," +
            " comment_flag.commentId as flag_commentId," +
            " comment_flag.flag as flag_flag," +
            " comment_flag.localFlag as flag_localFlag" +
            " from comment, comment_flag" +
            " where comment.referenceType = :referenceType" +
            " and comment.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " and comment.referenceId = :referenceId" +
            " and comment.parentId = :parentId" +
            " and comment.commentId = comment_flag.commentId" +
            " order by case when :isSortByCreatedACS = 1 then comment.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then comment.createdAt end desc")
    abstract DataSource.Factory<Integer, CommentEntity> getAllByParentIdImpl(String referenceType,
                                                                             String referenceId,
                                                                             String parentId,
                                                                             Boolean isDeleted,
                                                                             Boolean isSortByCreatedACS);

    @Query("SELECT comment.*," +
            " comment_flag.commentId as flag_commentId," +
            " comment_flag.flag as flag_flag," +
            " comment_flag.localFlag as flag_localFlag" +
            " from comment, comment_flag" +
            " where comment.referenceType = :referenceType" +
            " and comment.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " and comment.referenceId = :referenceId" +
            " and comment.parentId is null" +
            " and comment.commentId = comment_flag.commentId" +
            " order by case when :isSortByCreatedACS = 1 then comment.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then comment.createdAt end desc")
    abstract DataSource.Factory<Integer, CommentEntity> getAllByParentIdImpl(String referenceType,
                                                                             String referenceId,
                                                                             Boolean isDeleted,
                                                                             Boolean isSortByCreatedACS);


    @Query("SELECT *" +
            " from comment, amity_paging_id" +
            " where comment.referenceType = :referenceType" +
            " and comment.referenceId = :referenceId" +
            " and comment.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " and comment.parentId is null" +

            " and comment.updatedAt > :now" +
            " and comment.commentId not in " +

            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +

            " order by comment.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<CommentEntity> getLatestParentCommentImpl(String referenceType,
                                                                                   String referenceId,
                                                                                   Boolean isDeleted,
                                                                                   int hash,
                                                                                   int nonce,
                                                                                   DateTime now);

    @Query("SELECT *" +
            " from comment, amity_paging_id" +
            " where comment.referenceType = :referenceType" +
            " and comment.referenceId = :referenceId" +
            " and comment.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " and comment.parentId = :parentId" +

            " and comment.updatedAt > :now" +
            " and comment.commentId not in " +

            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +

            " order by comment.createdAt  desc" +
            " limit 1"
    )
    abstract Flowable<CommentEntity> getLatestChildCommentImpl(String referenceType,
                                                                                   String referenceId,
                                                                                   String parentId,
                                                                                   Boolean isDeleted,
                                                                                   int hash,
                                                                                   int nonce,
                                                                                   DateTime now);

    public Flowable<CommentEntity> getLatestComment(String referenceType,
                                                    String referenceId,
                                                    @Nullable String parentId,
                                                    Boolean isDeleted,
                                                    int hash,
                                                    int nonce) {
        if (parentId != null) {
            return getLatestChildCommentImpl(referenceType, referenceId, parentId,
                    isDeleted, hash, nonce, DateTime.now());
        } else {
            return getLatestParentCommentImpl(referenceType, referenceId,
                    isDeleted, hash, nonce, DateTime.now());
        }
    }


    public DataSource.Factory<Integer, CommentEntity> getAllByParentId(String referenceType,
                                                                       String referenceId,
                                                                       String parentId,
                                                                       Boolean isDeleted,
                                                                       AmityCommentSortOption sortOption) {
        if (parentId != null) {
            return getAllByParentIdImpl(referenceType, referenceId, parentId, isDeleted, isSortByCreatedACS(sortOption));
        } else {
            return getAllByParentIdImpl(referenceType, referenceId, isDeleted, isSortByCreatedACS(sortOption));
        }
    }

    @Query("SELECT comment.*," +
            " comment_flag.commentId as flag_commentId," +
            " comment_flag.flag as flag_flag," +
            " comment_flag.localFlag as flag_localFlag" +
            " from comment, comment_flag" +
            " where comment.referenceType = :referenceType" +
            " and comment.referenceId = :referenceId" +
            " and comment.commentId = comment_flag.commentId" +
            " and comment.isDeleted = 0" +
            " and comment.parentId is null" +
            " order by createdAt DESC" +
            " LIMIT 5")
    abstract List<CommentEntity> getLatestCommentsImpl(String referenceType, String referenceId);

    public List<CommentEntity> getLatestComments(String referenceType, String referenceId) {
        return getLatestCommentsImpl(referenceType, referenceId);
    }

    @Query("SELECT comment.*," +
            " comment_flag.commentId as flag_commentId," +
            " comment_flag.flag as flag_flag," +
            " comment_flag.localFlag as flag_localFlag" +
            " from comment, comment_flag" +
            " where comment.parentId = :parentId" +
            " and comment.referenceId = :referenceId" +
            " and comment.referenceType = :referenceType" +
            " and comment.commentId = comment_flag.commentId" +
            " order by case when :isSortByCreatedACS = 1 then comment.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then comment.createdAt end desc" +
            " LIMIT 5")
    abstract List<CommentEntity> getLatestRepliesImpl(String parentId, String referenceType, String referenceId, boolean isSortByCreatedACS);

    public List<CommentEntity> getLatestReplies(String parentId, String referenceType, String referenceId, AmityCommentSortOption sortBy) {
        return getLatestRepliesImpl(parentId, referenceType, referenceId, isSortByCreatedACS(sortBy));
    }

    @Query("SELECT *" +
            " from comment" +
            " where comment.referenceType = :referenceType" +
            " and comment.referenceId = :referenceId" +
            " and comment.createdAt > :offsetCommentTime" +
            " order by case when :isSortByCreatedACS = 1 then comment.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then comment.createdAt end desc")
    abstract Flowable<List<CommentEntity>> observeCommentAfterImpl(String referenceType, String referenceId, boolean isSortByCreatedACS, DateTime offsetCommentTime);

    @Query("SELECT *" +
            " from comment" +
            " where comment.referenceType = :referenceType" +
            " and comment.referenceId = :referenceId" +
            " and comment.parentId = :parentId" +
            " and comment.createdAt > :offsetCommentTime" +
            " order by case when :isSortByCreatedACS = 1 then comment.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then comment.createdAt end desc")
    abstract Flowable<List<CommentEntity>> observeCommentAfterFilterByParentImpl(String referenceType, String referenceId, String parentId, boolean isSortByCreatedACS, DateTime offsetCommentTime);

    public Flowable<List<CommentEntity>> observeCommentAfter(String referenceType,
                                                             String referenceId,
                                                             String parentId,
                                                             boolean isFilterByParentId,
                                                             AmityCommentSortOption sortOption,
                                                             String offsetCommentId) {
        if (isFilterByParentId) {
            return observeCommentAfterFilterByParentImpl(referenceType, referenceId, parentId, isSortByCreatedACS(sortOption), getCommentOffsetTime(offsetCommentId));
        } else {
            return observeCommentAfterImpl(referenceType, referenceId, isSortByCreatedACS(sortOption), getCommentOffsetTime(offsetCommentId));
        }
    }

    @Deprecated
    @Query("SELECT createdAt from comment where commentId = :commentId")
    public abstract DateTime getCommentCreatedTime(String commentId);

    private DateTime getCommentOffsetTime(String commentId) {
        if (commentId == null) {
            return DateTime.now();
        } else {
            return getCommentCreatedTime(commentId);
        }
    }

    @Query("SELECT *" +
            " from comment" +
            " where comment.commentId = :id" +
            " LIMIT 1")
    abstract CommentEntity getByIdNowImpl(String id);

    @Override
    public CommentEntity getByIdNow(@NonNull String id) {
        return getByIdNowImpl(id);
    }

    @Query("SELECT *" +
            " from comment" +
            " where comment.commentId IN (:ids)")
    abstract List<CommentEntity> getByIdsNowImpl(List<String> ids);

    @NonNull
    @Override
    public List<CommentEntity> getByIdsNow(@NonNull List<String> ids) {
        return getByIdsNowImpl(ids);
    }

    @Query("SELECT comment.*," +
            " comment_flag.commentId as flag_commentId," +
            " comment_flag.flag as flag_flag," +
            " comment_flag.localFlag as flag_localFlag" +
            " from comment, comment_flag" +
            " where comment.commentId = :commentId" +
            " and comment.commentId = comment_flag.commentId" +
            " LIMIT 1")
    abstract Flowable<CommentEntity> getByIdImpl(String commentId);

    public Flowable<CommentEntity> getById(String commentId) {
        return getByIdImpl(commentId);
    }

    @Query("UPDATE comment set syncState = 'failed' where syncState = 'syncing'")
    public abstract void initSyncStateOnStartup();

    @Query("DELETE from comment")
    public abstract void deleteAll();

    @Query("UPDATE comment set isDeleted = 1 where commentId = :commentId")
    public abstract Completable softDeleteById(String commentId);

    @Query("DELETE from comment where commentId = :commentId")
    public abstract Completable deleteById(String commentId);

    @Query("DELETE from comment where referenceId = :referenceId")
    public abstract Completable deleteByReferenceId(String referenceId);

    private Boolean isSortByCreatedACS(AmityCommentSortOption sortBy) {
        return sortBy == AmityCommentSortOption.FIRST_CREATED;
    }

    // dummy update post
    @Query("UPDATE comment set commentId = commentId where commentId = :commentId")
    public abstract void updateComment(String commentId);

    public void markAllDeletedBeforeCommentId(String commentId) {
        CommentEntity comment = getByIdNow(commentId);
        if(comment != null) {
            markAllDeletedBeforeCreatedAt(comment.getCreatedAt());
        }
    }

    public  void markAllDeletedAfterCommentId(String commentId){
        CommentEntity comment = getByIdNow(commentId);
        if(comment != null) {
            markAllDeletedAfterCreatedAt(comment.getCreatedAt());
        }
    }

    @Query("UPDATE comment set isDeleted = 1 where createdAt < :createdAt")
    public abstract void markAllDeletedBeforeCreatedAt(DateTime createdAt);

    @Query("UPDATE comment set isDeleted = 1 where createdAt > :createdAt")
    public abstract void markAllDeletedAfterCreatedAt(DateTime createdAt);

    public Flowable<List<CommentEntity>> getByCommentIds(List<String> ids) {
        return getByCommentIdsImpl(ids);
    }

    @Query("SELECT *" +
            " from comment" +
            " where comment.commentId IN (:commentIds)")
    abstract Flowable<List<CommentEntity>> getByCommentIdsImpl(List<String> commentIds);

}
