package com.amity.socialcloud.sdk.core.data.presence

import com.amity.socialcloud.sdk.core.data.presence.user.UserPresenceEntityMapper
import com.amity.socialcloud.sdk.core.data.presence.user.UserPresenceModelMapper
import com.amity.socialcloud.sdk.core.data.presence.user.UserPresencePersister
import com.amity.socialcloud.sdk.core.presence.PresenceConfig
import com.amity.socialcloud.sdk.model.core.presence.AmityUserPresence
import com.ekoapp.ekosdk.internal.api.dto.UserPresenceListDto
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single

internal class PresenceRepository {

    fun getPresenceUserSetting(): Single<Boolean> {
        return PresenceRemoteDataStore().getPresenceSetting()
            .flatMap { dto ->
                dto.user.enabled?.let { Single.just(it) } ?: Single.never()
            }
    }

    fun getPresenceNetworkSetting(): Single<PresenceConfig> {
        return PresenceRemoteDataStore().getPresenceSetting()
            .map {
                PresenceConfig(
                    isSyncEnabled = it.user.enabled ?: false,
                    heartbeatSyncInterval = it.network.heartbeatInterval ?: -1,
                    presenceSyncInterval = it.network.presenceSyncInterval ?: -1
                )
            }
    }

    fun updatePresenceUserSetting(enable: Boolean): Completable {
        return PresenceRemoteDataStore().updatePresenceUserSetting(enable)
            .ignoreElement()
    }

    fun sendPresenceHeartbeat(): Completable {
        return PresenceRemoteDataStore().sendPresenceHeartbeat()
            .ignoreElement()
    }

    fun getUserPresences(userIds: List<String>): Single<List<AmityUserPresence>> {
        return PresenceRemoteDataStore().getUserPresence(userIds)
            .flatMap { persistAndReturnUserPresences(it) }
    }

    fun getUserPresence(userId: String): AmityUserPresence? {
        return PresenceLocalDataStore().getUserPresence(userId)?.let {
            UserPresenceModelMapper().map(it)
        }
    }
    
    fun getUserPresenceNow(userId: String): AmityUserPresence? {
        return PresenceLocalDataStore().getUserPresenceNow(userId)?.let {
            UserPresenceModelMapper().map(it)
        }
    }

    fun getOnlineUsers(): Single<List<AmityUserPresence>> {
        return PresenceRemoteDataStore().getOnlineUsers()
            .flatMap { persistAndReturnUserPresences(it) }
    }

    fun getOnlineUsersCount(): Single<Int> {
        return PresenceRemoteDataStore().getOnlineUsersCount()
            .map { it["count"].asInt }
    }

    private fun persistAndReturnUserPresences(dto: UserPresenceListDto): Single<List<AmityUserPresence>> {
        return UserPresencePersister().persist(dto)
            .toSingle {
                UserPresenceEntityMapper().map(dto.presences).map { entity ->
                    UserPresenceModelMapper().map(entity)
                }
            }
    }

}