package com.amity.socialcloud.sdk.model.social.community

import android.os.Parcelable
import com.amity.socialcloud.sdk.api.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.api.social.post.review.AmityReviewStatus
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.model.core.events.AmityCommunityEvents
import com.amity.socialcloud.sdk.model.core.events.AmityTopic
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.model.social.category.AmityCommunityCategory
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.ekoapp.ekosdk.EkoCategoryIds
import com.ekoapp.ekosdk.ReactorObject
import com.amity.socialcloud.sdk.social.domain.community.CommunityPostCountUseCase
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Flowable
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.amity.types.ObjectId
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityCommunity internal constructor(
    private val communityId: String = ObjectId.get().toHexString(),
    private val channelId: String,
    private val userId: String,
    private val displayName: String,
    private val description: String,
    private val avatarFileId: String,
    private val isOfficial: Boolean,
    private val isPublic: Boolean,
    private val onlyAdminCanPost: Boolean,
    private val allowCommentInStory: Boolean,
    private val tags: AmityTags = AmityTags(),
    private val metadata: JsonObject?,
    private val postsCount: Int = 0,
    private val membersCount: Int = 0,
    private val isJoined: Boolean,
    private val isDeleted: Boolean,
    private val categoryIds: EkoCategoryIds,
    internal var categories: List<AmityCommunityCategory>,
    internal var user: AmityUser?,
    internal var avatarImage: AmityImage?,
    private val isPostReviewEnabled: Boolean,
    private val createdAt: DateTime,
    private val updatedAt: DateTime,
    internal val path: String
) : Parcelable, ReactorObject {

    fun getCommunityId(): String {
        return communityId
    }

    fun getChannelId(): String {
        return channelId
    }

    fun getCreatorId(): String {
        return userId
    }

    fun getDisplayName(): String {
        return displayName
    }

    fun getDescription(): String {
        return description
    }

    internal fun getAvatarFileId(): String {
        return avatarFileId
    }

    fun isOfficial(): Boolean {
        return isOfficial
    }

    fun isPublic(): Boolean {
        return isPublic
    }

    fun getTags(): AmityTags {
        return tags
    }

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getPostCount(): Int {
        return postsCount
    }

    fun getMemberCount(): Int {
        return membersCount
    }

    fun isJoined(): Boolean {
        return isJoined
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getCreator(): AmityUser? {
        return user
    }

    fun getAvatar(): AmityImage? {
        return avatarImage
    }

    fun getCategoryIds(): EkoCategoryIds {
        return categoryIds
    }

    fun getCategories(): List<AmityCommunityCategory> {
        return categories
    }

    @Deprecated("Use getPostCount(reviewStatus: AmityReviewStatus) instead")
    fun getPostCount(feedType: AmityFeedType): Flowable<Int> {
        return CommunityPostCountUseCase().execute(communityId, feedType)
    }

    fun getPostCount(reviewStatus: AmityReviewStatus): Flowable<Int> {
        return CommunityPostCountUseCase().execute(communityId, AmityFeedType.enumOf(reviewStatus.apiKey))
    }

    fun getCreatedAt(): DateTime {
        return createdAt
    }

    fun getUpdatedAt(): DateTime {
        return updatedAt
    }

    override fun updatedAt(): DateTime {
        return updatedAt
    }

    override fun uniqueId(): String {
        return communityId
    }

    fun getPostSettings(): AmityCommunityPostSettings {
        return when {
            !onlyAdminCanPost && isPostReviewEnabled -> AmityCommunityPostSettings.ADMIN_REVIEW_POST_REQUIRED
            !onlyAdminCanPost && !isPostReviewEnabled -> AmityCommunityPostSettings.ANYONE_CAN_POST
            else -> AmityCommunityPostSettings.ADMIN_CAN_POST_ONLY
        }
    }

    fun getStorySettings() : AmityCommunityStorySettings {
        return AmityCommunityStorySettings(allowCommentInStory)
    }

    fun subscription(events: AmityCommunityEvents): AmityTopicSubscription {
        return AmityTopicSubscription(AmityTopic.COMMUNITY(this, events))
    }

}