package com.ekoapp.ekosdk.internal.api

import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityError.Companion.from
import com.amity.socialcloud.sdk.model.core.error.AmityException.Companion.create
import com.amity.socialcloud.sdk.model.core.session.AmityGlobalBanEvent
import com.ekoapp.ekosdk.internal.api.EkoEndpoint.getSocketUrl
import com.ekoapp.ekosdk.internal.api.event.SocketEventListener
import com.ekoapp.ekosdk.internal.api.event.StreamDidFlagListener
import com.ekoapp.ekosdk.internal.api.event.StreamDidStartListener
import com.ekoapp.ekosdk.internal.api.event.StreamDidStopListener
import com.ekoapp.ekosdk.internal.api.event.StreamDidTerminateListener
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.ekoapp.ekosdk.sdk.BuildConfig
import com.google.common.base.Objects
import com.google.common.collect.ImmutableSet
import com.google.gson.JsonParser
import com.jakewharton.rxrelay3.BehaviorRelay
import io.reactivex.rxjava3.core.BackpressureStrategy
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.rxjava3.subjects.PublishSubject
import io.socket.client.IO
import io.socket.client.Manager
import io.socket.client.Socket
import io.socket.engineio.client.Transport
import io.socket.engineio.client.transports.WebSocket
import okhttp3.Dispatcher
import okhttp3.OkHttpClient
import java.net.URISyntaxException
import java.util.*
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicInteger


@Deprecated("")
class EkoSocket(
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
    override fun onSessionStateChange(sessionState: SessionState) {}

    init {
        companionSessionStateEventBus = getSessionStateEventBus()
    }

    override fun establish(account: EkoAccount) {
        terminateSocket()
        currentSocket = init(account)
        currentSocket.connect()
    }
    override fun destroy() {
        terminateSocket()
    }

    override fun handleTokenExpire() {
        terminateSocket()
    }

    fun disconnect() {
        terminateSocket()
    }

    //to properly close a client, followed by socket.io best practice
    //https://socketio.github.io/socket.io-client-java/faq.html#How_to_properly_close_a_client
    private fun terminateSocket() {
        if (currentSocket.connected()) {
            currentSocket.disconnect()
        }
        currentDispatcher.executorService.shutdown()
        // FIXME: remove logs
        val socketHash = Integer.toHexString(currentSocket.hashCode())
        AmityLog.e("socket", "terminate socket: $socketHash" )
    }


    companion object {
        private val TAG = EkoSocket::class.java.name
        private val PROXY = Socket(null, null, null)
        private val rpcId = AtomicInteger(0)
        private var currentAccount = EkoAccount.create("seed")
        private var currentSocket: Socket = PROXY
        private var currentDispatcher: Dispatcher = Dispatcher()
        private val connectionEventRelay = BehaviorRelay.create<SocketConnectionEvent>()
        private val globalBanEventPublisher = PublishSubject.create<AmityGlobalBanEvent>()
        private lateinit var companionSessionStateEventBus: SessionStateEventBus

        @Throws(URISyntaxException::class)
        private fun init(account: EkoAccount): Socket {
            currentDispatcher = Dispatcher()
            currentAccount = account
            val userId = account.userId
            val authority = getSocketUrl()
            AmityLog.i(TAG,"init new socket for: %s , url: %s", userId, authority)
            /*
            Check default value at  io.socket.client.Manager
            reconnectionDelay = 1000
            reconnectionDelayMax = 5000
            randomizationFactor = 0.5
         */

            val okHttpClient: OkHttpClient = OkHttpClient.Builder()
                .dispatcher(currentDispatcher)
                .readTimeout(1, TimeUnit.MINUTES) // important for HTTP long-polling
                .build()
            val options = IO.Options()
            options.callFactory = okHttpClient;
            options.webSocketFactory = okHttpClient;
            options.reconnectionDelayMax = 10000
            options.transports = arrayOf(WebSocket.NAME)
            val socket = IO.socket(authority, options)
            val events: Set<String> = ImmutableSet.builder<String>()
                .add(Socket.EVENT_CONNECT)
                .add(Socket.EVENT_CONNECT_ERROR)
                .add(Socket.EVENT_CONNECT_TIMEOUT)
                .add(Socket.EVENT_CONNECTING)
                .add(Socket.EVENT_DISCONNECT)
                .add(Socket.EVENT_ERROR)
                .add(Socket.EVENT_RECONNECT)
                .add(Socket.EVENT_RECONNECT_ATTEMPT)
                .add(Socket.EVENT_RECONNECT_FAILED)
                .add(Socket.EVENT_RECONNECTING)
                .add(Socket.EVENT_PING)
                .add(Socket.EVENT_PONG)
                .add(Socket.EVENT_MESSAGE)
                .build()

            socket.io().on(Manager.EVENT_TRANSPORT) { it ->
                val transport = it[0] as Transport
                transport.on(Transport.EVENT_REQUEST_HEADERS) { args ->
                    val headers = args[0] as MutableMap<String, List<String>>
                    headers["X-ACCESS-TOKEN"] = listOf(account.accessToken)
                }
            }

            for (event in events) {
                socket.on(event) { args: Array<Any>? ->
                    // FIXME doesn't look good. Find other way?
                    if (Objects.equal(socket, currentSocket)) {
                        val sce = SocketConnectionEvent(userId, socket, event, args!!)
                        this.connectionEventRelay.accept(sce)
                    }
                    if (BuildConfig.DEBUG) {
                        val socketHash = Integer.toHexString(socket.hashCode())
                        AmityLog.tag(TAG).e(
                            "socket: %s uid: %s connected: %s event: %s args: %s",
                            socketHash,
                            userId,
                            socket.connected(),
                            event,
                            Arrays.deepToString(args)
                        )
                    }
                }
            }


            socket.on(Socket.EVENT_DISCONNECT) { args: Array<Any?> ->
                if (args.size > 0 && Objects.equal(
                        args[0], "io server disconnect"
                    )
                ) {
                    socket.connect()
                }
            }
            socket.on(Socket.EVENT_ERROR) { args: Array<Any?>? ->
                try {
                    val parser = JsonParser()
                    val element = parser.parse(Arrays.deepToString(args))
                    val array = element.asJsonArray
                    val `object` = array[0].asJsonObject
                    val exception = create(
                        `object`["message"].asString,
                        null,
                        `object`["code"].asInt
                    )
                    if (from(exception).`is`(AmityError.USER_IS_GLOBAL_BANNED)) {
                        globalBanEventPublisher.onNext(AmityGlobalBanEvent(userId))
                        companionSessionStateEventBus.publish(SessionState.Terminated(exception))
                    }
                } catch (e: Exception) {
                    AmityLog.tag(TAG).e(e, String.format("event: error arg: %s", Arrays.deepToString(args)))
                }
            }
            Completable.fromAction {
                val channelDao = UserDatabase.get().channelDao()
                channelDao.deleteAllLocallyInactiveChannelsAndUpdateAllActiveChannelsToNotReading()
            }
                .subscribeOn(Schedulers.io())
                .subscribe()
            subscribeSocketEvent(socket, StreamDidStartListener())
            subscribeSocketEvent(socket, StreamDidStopListener())
            subscribeSocketEvent(socket, StreamDidFlagListener())
            subscribeSocketEvent(socket, StreamDidTerminateListener())
            return socket
        }

        private fun subscribeSocketEvent(socket: Socket, listener: SocketEventListener) {
            socket.on(listener.event, listener)
        }

        @JvmStatic
        fun connectionEvent(): Flowable<SocketConnectionEvent> {
            return this.connectionEventRelay.toFlowable(BackpressureStrategy.BUFFER)
        }

        val globalBanEvents: Flowable<AmityGlobalBanEvent>
            get() = globalBanEventPublisher.toFlowable(BackpressureStrategy.BUFFER)
    }

}