package com.amity.socialcloud.sdk.chat.data.marker.subchannel

import com.amity.socialcloud.sdk.chat.data.marker.reader.MarkReadRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.marker.reader.SubChannelReadingPersister
import com.amity.socialcloud.sdk.chat.domain.marker.channel.ReCalculateChannelUnreadInfoUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.subchannel.GetSubChannelUnreadCountUseCase
import com.amity.socialcloud.sdk.core.CoreClient
import com.ekoapp.ekosdk.internal.SubChannelUnreadInfoEntity
import com.ekoapp.ekosdk.internal.UserSubChannelMarkerEntity
import io.reactivex.rxjava3.core.Completable

internal class SubChannelMarkerRepository {
    
    fun fetchSubChannelMarker(subChannelIds: List<String>): Completable {
        return if (!CoreClient.isUnreadCountEnable() || subChannelIds.isEmpty()) {
            Completable.complete()
        } else {
            SubChannelMarkerRemoteDataStore()
                .querySubChannelMarkers(subChannelIds)
                .flatMapCompletable {
                    if (CoreClient.isConsistentMode()) {
                        SubChannelUnreadInfoQueryPersister().persist(it)
                    } else {
                        SubChannelMarkerQueryPersister().persist(it)
                    }
                }
        }
    }
    
    fun getSubChannelMarker(subChannelId: String): UserSubChannelMarkerEntity? {
        return GetSubChannelUnreadCountUseCase().execute(subChannelId)
    }

    fun updateReadToSegment(subChannelId: String, segment: Int){
        SubChannelMarkerLocalDataStore().updateReadToSegment(subChannelId,segment)
    }
    
    fun getUserSubChannelMarker(subChannelId: String): UserSubChannelMarkerEntity? {
        return SubChannelMarkerLocalDataStore().getUserSubChannelMarker(subChannelId)
    }
    
    fun startSubChannelReading(subChannelId: String): Completable {
        return MarkReadRemoteDataStore().startSubChannelReading(subChannelId)
            .flatMapCompletable {
                SubChannelReadingPersister().persist(it)
            }
            .subscribeOn(io.reactivex.rxjava3.schedulers.Schedulers.io())
    }
    
    fun readSubChannels(subChannelIds: List<String>): Completable {
        return MarkReadRemoteDataStore().readSubChannel(subChannelIds)
                .flatMapCompletable {
                    SubChannelReadingPersister().persist(it)
                }
                .subscribeOn(io.reactivex.rxjava3.schedulers.Schedulers.io())
    }
    
    fun stopSubChannelReading(subChannelId: String): Completable {
        return MarkReadRemoteDataStore().stopSubChannelReading(subChannelId)
            .flatMapCompletable {
                SubChannelReadingPersister().persist(it)
            }
            .subscribeOn(io.reactivex.rxjava3.schedulers.Schedulers.io())
    }
    
    fun fetchSubChannelUnreadInfo(channelIds: List<String>): Completable {
        return if (CoreClient.isUnreadCountEnable()) {
            SubChannelMarkerRemoteDataStore().queryUserSubChannelMarkers(channelIds)
                    .flatMapCompletable {
                        SubChannelUnreadInfoQueryPersister().persist(it)
                    }
                    .subscribeOn(io.reactivex.rxjava3.schedulers.Schedulers.io())
        } else {
            Completable.complete()
        }
    }
    
    fun getSubChannelUnreadInfo(subChannelId: String): SubChannelUnreadInfoEntity? {
        return SubChannelMarkerLocalDataStore().getSubChannelUnreadInfo(subChannelId)
    }
    
    fun saveSubChannelUnreadInfo(unreadInfo: List<SubChannelUnreadInfoEntity>) {
        val deletedSubChannels: List<String> = unreadInfo
            .map { it.subChannelId }
            .let(SubChannelMarkerLocalDataStore()::getDeletedSubChannelIds)
        unreadInfo
            .filter { !deletedSubChannels.contains(it.subChannelId) }
            .let(SubChannelMarkerLocalDataStore()::saveSubChannelUnreadInfo)
        unreadInfo
            .map { it.channelId }
            .distinct()
            .forEach { channelId ->
                ReCalculateChannelUnreadInfoUseCase().execute(channelId)
            }
    }
    
    fun deleteUnreadInfoBySubChannelId(subChannelId: String) {
        SubChannelMarkerLocalDataStore().deleteUnreadInfoBySubChannelId(subChannelId)
    }
    
    fun deleteUnreadInfoByChannelId(channelId: String) {
        SubChannelMarkerLocalDataStore().deleteUnreadInfoByChannelId(channelId)
    }
    
    fun deleteUnreadInfoCacheByChannelId(channelId: String) {
        SubChannelMarkerLocalDataStore().deleteUnreadInfoCacheByChannelId(channelId)
    }
}