package com.ekoapp.ekosdk.internal.mediator

import co.amity.rxbridge.toRx3
import co.amity.rxremotemediator.AmityPagingId
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

abstract class SinglePageMediator<Entity : Any, ResponseDto : Any>(
    val nonce: Int,
    var dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator
) {
	private var highestPosition = 0
	private var lowestPosition = 0
	
	abstract fun getRequest(): Single<ResponseDto>
	abstract fun persistResponse(dto: ResponseDto): Completable
	abstract fun convertResponseToQueryToken(dto: ResponseDto): QueryStreamQueryToken
	
	fun load(): Completable {
		val tokenDao = UserDatabase.get().queryTokenDao()
		return fetch()
			.flatMap { token ->
				tokenDao.clearPagingIds(dynamicQueryStreamKeyCreator.toMap(), nonce)
					.toRx3()
					.andThen(Single.defer { Single.just(token) })
			}
			.subscribeOn(Schedulers.io())
			.map {
				it.apply {
					this.nonce = this@SinglePageMediator.nonce
				}
			}.flatMapCompletable {
				insertToken(it).toRx3()
			}
	}
	
	private fun fetch(): Single<QueryStreamQueryToken> {
		return getRequest()
			.flatMap {
				onFirstPageLoaded(it)
				persistResponse(it).andThen(Single.just(it))
			}
			.map(::convertResponseToQueryToken)
	}
	
	open fun onFirstPageLoaded(dto: ResponseDto) {
	
	}
	
	private fun insertToken(token: QueryStreamQueryToken): io.reactivex.Completable {
		val tokenDao = UserDatabase.get().queryTokenDao()
		return tokenDao.insertPagingIds(token.primaryKeys.mapIndexed { index, id ->
			AmityPagingId(queryParameters = dynamicQueryStreamKeyCreator.toMap(), id = id)
				.apply {
					this.nonce = this@SinglePageMediator.nonce
					this.position = ++highestPosition
				}
		})
	}
	
}