package com.ekoapp.ekosdk.internal.data.dao

import androidx.room.Dao
import androidx.room.Query
import androidx.room.Transaction
import com.amity.socialcloud.sdk.api.chat.message.query.AmityMessageQuerySortOption
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.EkoTagDao.Companion.update
import com.ekoapp.ekosdk.internal.data.model.EkoMessageTag.Companion.factory
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import org.joda.time.DateTime

@Dao
abstract class EkoMessageDao internal constructor() : EkoObjectDao<EkoMessageEntity>(), AmityPagingDao<EkoMessageEntity> {
	private val messageTagDao: EkoMessageTagDao
	
	init {
		messageTagDao = UserDatabase.get().messageTagDao()
	}
	
	@Query("SELECT message.*" +
			" from message" +
			" where message.subChannelId = :subChannelId" +
			" and message.isDeleted = case when :isDeleted is not null then :isDeleted else isDeleted end" +
			" order by channelSegment DESC LIMIT 1")
	abstract fun getLatestMessageImpl(subChannelId: String, isDeleted: Boolean?): Flowable<EkoMessageEntity>
	fun getLatestMessage(channelId: String): Flowable<EkoMessageEntity> {
		return getLatestMessageImpl(channelId, null)
	}
	
	fun getLatestMessage(channelId: String, isDeleted: Boolean?): Flowable<EkoMessageEntity> {
		return getLatestMessageImpl(channelId, isDeleted)
	}
	
	@Query("SELECT channelSegment from message where subChannelId = :subChannelId and syncState = 'synced' order by channelSegment DESC LIMIT 1")
	abstract fun getHighestSegment(subChannelId: String): Int
	@Query("UPDATE message set syncState = 'failed' where syncState = 'syncing'")
	abstract fun cleanUpSyncingStateOnStartup()
	@Query("UPDATE message set syncState = 'failed' where syncState = 'uploading'")
	abstract fun cleanUpUploadingStateOnStartup()
	@Query("DELETE from message where syncState = 'failed'")
	abstract fun cleanUpFailedMessages()
	@Query("UPDATE message set syncState = :state where messageId = :messageId")
	abstract fun updateSyncState(messageId: String, state: String): Completable
	@Query("DELETE from message")
	abstract override fun deleteAll()
	@Query("DELETE from message where channelId = :channelId")
	abstract fun deleteAllFromChannel(channelId: String)
	@Query("SELECT * from message where channelId = :channelId order by createdAt DESC LIMIT -1 OFFSET :offset")
	abstract fun getOldMessages(channelId: String, offset: Int): List<EkoMessageEntity>
	@Transaction
	open fun retainLatestFromChannel(channelId: String, offset: Int) {
		val oldMessages = getOldMessages(channelId, offset)
		delete(oldMessages)
	}
	
	@Query("DELETE from message where messageId = :messageId")
	abstract fun hardDeleteMessage(messageId: String): Completable
	@Query("DELETE from message where channelId = :channelId")
	abstract fun hardDeleteAllFromChannel(channelId: String)
	@Query("UPDATE message set isDeleted = 1, data = null where channelId = :channelId and userId = :userId")
	abstract fun softDeleteFromChannelByUserIdImpl(channelId: String, userId: String)
	@Transaction
	open fun softDeleteFromChannelByUserId(channelId: String, userId: String) {
		softDeleteFromChannelByUserIdImpl(channelId, userId)
	}
	
	@Query("UPDATE message set userId = :userId where userId = :userId")
	abstract fun updateUserImpl(userId: String)
	
	@Transaction //dummy update, for triggering all messages sent by this userId.
	open fun updateUser(userId: String) {
		updateUserImpl(userId)
	}
	
	@Transaction
	override fun insert(message: EkoMessageEntity) {
		super.insert(message)
		update(message, messageTagDao, factory)
	}
	
	@Transaction
	override fun insert(messages: List<EkoMessageEntity>) {
		super.insert(messages)
		update(messages, messageTagDao, factory)
	}
	
	@Transaction
	override fun update(message: EkoMessageEntity) {
		super.update(message)
		update(message, messageTagDao, factory)
	}
	
	@Query("SELECT message.*" +
			" from message" +
			" where message.messageId = :messageId" +
			" LIMIT 1")
	abstract fun getByIdImpl(messageId: String): Flowable<EkoMessageEntity>
	fun getById(messageId: String): Flowable<EkoMessageEntity> {
		return getByIdImpl(messageId)
	}
	
	@Query("SELECT message.uniqueId" +
			" from message" +
			" where message.messageId = :messageId" +
			" LIMIT 1")
	abstract fun getUniqueIdByMessageIdImpl(messageId: String): String
	
	@Query("SELECT message.*" +
			" from message" +
			" where message.uniqueId = :uniqueId" +
			" LIMIT 1")
	abstract fun getByIdNowImpl(uniqueId: String): EkoMessageEntity?
	override fun getByIdNow(uniqueId: String): EkoMessageEntity? {
		return getByIdNowImpl(uniqueId)
	}
	
	@Query("SELECT message.*" +
			" from message" +
			" where message.messageId = :messageId" +
			" LIMIT 1")
	abstract fun getByMessageIdNowImpl(messageId: String): EkoMessageEntity?
	fun getUniqueIdByMessageId(messageId: String): String? {
		return getUniqueIdByMessageIdImpl(messageId)
	}
	
	fun getByMessageIdNow(messageId: String): EkoMessageEntity? {
		return getByMessageIdNowImpl(messageId)
	}
	
	@Query("SELECT message.*" +
			" from message" +
			" where message.uniqueId IN (:uniqueIds)")
	abstract fun getByIdsNowImpl(uniqueIds: List<String>): List<EkoMessageEntity>
	override fun getByIdsNow(ids: List<String>): List<EkoMessageEntity> {
		return getByIdsNowImpl(ids)
	}
	
	@Query("SELECT message.*" +
			" from message" +
			" where message.subChannelId = :subChannelId" +
			" and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
			" when :isFilterByParentId and :parentId is null then message.parentId is null " +
			" else message.messageId is not null end" +  // always true
			" and case when :isFilterByTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags)) " +
			" else message.messageId is not null end" +  // always true
			" and message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
			" and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
			" else message.messageId is not null end" +  // always true
			" and message.type IN (:type)" +
			" order by" +
			" case when :isSortedAsc = 1 then message.createdAt end asc," +
			" case when :isSortedAsc = 0 then message.createdAt end desc")
	abstract fun observeMessagesWithTypeImpl(subChannelId: String,
	                                         isFilterByParentId: Boolean,
	                                         parentId: String?,
	                                         isFilterByTags: Boolean,
	                                         includingTags: Array<String>?,
	                                         excludingTags: Array<String?>?,
	                                         isDeleted: Boolean?,
	                                         type: List<String?>?,
	                                         isSortedAsc: Boolean?): Flowable<List<EkoMessageEntity>>
	
	@Query("SELECT message.*" +
			" from message" +
			" where message.subChannelId = :subChannelId" +
			" and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
			" when :isFilterByParentId and :parentId is null then message.parentId is null " +
			" else message.messageId is not null end" +  // always true
			" and case when :isFilterByTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags)) " +
			" else message.messageId is not null end" +  // always true
			" and message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
			" and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
			" else message.messageId is not null end" +  // always true
			" order by" +
			" case when :isSortedAsc = 1 then message.createdAt end asc," +
			" case when :isSortedAsc = 0 then message.createdAt end desc")
	abstract fun observeMessagesImpl(subChannelId: String?,
	                                 isFilterByParentId: Boolean,
	                                 parentId: String?,
	                                 isFilterByTags: Boolean,
	                                 includingTags: Array<String>,
	                                 excludingTags: Array<String>,
	                                 isDeleted: Boolean?,
	                                 isSortedAsc: Boolean): Flowable<List<EkoMessageEntity>>
	
	fun observeMessages(subChannelId: String,
	                    isFilterByParentId: Boolean,
	                    parentId: String?,
	                    includingTags: AmityTags,
	                    excludingTags: AmityTags,
	                    isDeleted: Boolean?,
	                    type: List<String>,
	                    sortOption: AmityMessageQuerySortOption): Flowable<List<EkoMessageEntity>> {
		return if (type.isEmpty()) {
			observeMessagesImpl(subChannelId,
					isFilterByParentId,
					parentId,
					!includingTags.isEmpty(),
					includingTags.toTypedArray(),
					excludingTags.toTypedArray(),
					isDeleted,
					sortOption === AmityMessageQuerySortOption.FIRST_CREATED)
		} else {
			observeMessagesWithTypeImpl(subChannelId,
					isFilterByParentId,
					parentId,
					!includingTags.isEmpty(),
					includingTags.toTypedArray(),
					excludingTags.toTypedArray(),
					isDeleted,
					type,
					sortOption === AmityMessageQuerySortOption.FIRST_CREATED)
		}
	}
	
	@Query("SELECT *" +
			" from message" +
			" where message.subChannelId = :subChannelId" +
			" and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
			" when :isFilterByParentId and :parentId is null then message.parentId is null " +
			" else message.messageId is not null end" +  // always true
			" and case when :isIncludingTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags))" +
			" else message.messageId is not null end" +  // always true
			" and case when :isExcludingTags then message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
			" else message.messageId is not null end" +  // always true
			" and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
			" else message.messageId is not null end" +  // always true
			" and message.updatedAt > :now" +
			" and message.uniqueId not in " +
			"(" +
			"SELECT amity_paging_id.id" +
			" from amity_paging_id" +
			" where amity_paging_id.hash = (:hash)" +
			" and amity_paging_id.nonce = (:nonce) " +
			")" +
			" order by message.updatedAt desc" +
			" limit 1")
	abstract fun getLatestMessageImpl(subChannelId: String,
	                                  isFilterByParentId: Boolean?,
	                                  parentId: String?,
	                                  isIncludingTags: Boolean,
	                                  includingTags: Array<String>,
	                                  isExcludingTags: Boolean,
	                                  excludingTags: Array<String>,
	                                  isDeleted: Boolean?,
	                                  hash: Int,
	                                  nonce: Int,
	                                  now: DateTime): Flowable<EkoMessageEntity>
	
	@Query("SELECT *" +
			" from message" +
			" where message.subChannelId = :subChannelId" +
			" and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
			" when :isFilterByParentId and :parentId is null then message.parentId is null " +
			" else message.messageId is not null end" +  // always true
			" and case when :isIncludingTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags))" +
			" else message.messageId is not null end" +  // always true
			" and case when :isExcludingTags then message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
			" else message.messageId is not null end" +  // always true
			" and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
			" else message.messageId is not null end" +  // always true
			" and message.updatedAt > :now" +
			" and message.uniqueId not in " +
			"(" +
			"SELECT amity_paging_id.id" +
			" from amity_paging_id" +
			" where amity_paging_id.hash = (:hash)" +
			" and amity_paging_id.nonce = (:nonce) " +
			")" +
			" and message.type in (:type)" +
			" order by message.updatedAt desc" +
			" limit 1")
	abstract fun getLatestMessageWithTypeImpl(subChannelId: String,
	                                          isFilterByParentId: Boolean?,
	                                          parentId: String?,
	                                          isIncludingTags: Boolean,
	                                          includingTags: Array<String>,
	                                          isExcludingTags: Boolean,
	                                          excludingTags: Array<String>,
	                                          isDeleted: Boolean?,
	                                          type: List<String>,
	                                          hash: Int,
	                                          nonce: Int,
	                                          now: DateTime): Flowable<EkoMessageEntity>
	
	fun getLatestMessage(subChannelId: String,
	                     isFilterByParentId: Boolean?,
	                     parentId: String?,
	                     includingTags: Array<String>,
	                     excludingTags: Array<String>,
	                     isDeleted: Boolean?,
	                     type: List<String>,
	                     hash: Int,
	                     nonce: Int,
	                     now: DateTime): Flowable<EkoMessageEntity> {
		return if (type.isEmpty()) {
			getLatestMessageImpl(subChannelId,
					isFilterByParentId,
					parentId,
					includingTags.size > 0,
					includingTags,
					excludingTags.size > 0,
					excludingTags,
					isDeleted,
					hash,
					nonce,
					now)
		} else {
			getLatestMessageWithTypeImpl(subChannelId,
					isFilterByParentId,
					parentId,
					includingTags.size > 0,
					includingTags,
					excludingTags.size > 0,
					excludingTags,
					isDeleted,
					type,
					hash,
					nonce,
					now)
		}
	}
	
	@Query("SELECT *" +
			" from message" +
			" where message.subChannelId = :subChannelId" +
			" and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
			" when :isFilterByParentId and :parentId is null then message.parentId is null " +
			" else message.messageId is not null end" +  // always true
			" and case when :isIncludingTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags))" +
			" else message.messageId is not null end" +  // always true
			" and case when :isExcludingTags then message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
			" else message.messageId is not null end" +  // always true
			" and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
			" else message.messageId is not null end" +  // always true
			" and message.syncState != 'synced' " +
			" and message.uniqueId not in " +
			"(" +
			"SELECT amity_paging_id.id" +
			" from amity_paging_id" +
			" where amity_paging_id.hash = (:hash)" +
			" and amity_paging_id.nonce = (:nonce) " +
			")" +
			" order by message.updatedAt desc" +
			" limit 1")
	abstract fun getLatestUnsyncMessageImpl(subChannelId: String,
	                                        isFilterByParentId: Boolean?,
	                                        parentId: String?,
	                                        isIncludingTags: Boolean,
	                                        includingTags: Array<String>,
	                                        isExcludingTags: Boolean,
	                                        excludingTags: Array<String>,
	                                        isDeleted: Boolean?,
	                                        hash: Int,
	                                        nonce: Int): Flowable<EkoMessageEntity>
	
	@Query("SELECT *" +
			" from message" +
			" where message.subChannelId = :subChannelId" +
			" and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
			" when :isFilterByParentId and :parentId is null then message.parentId is null " +
			" else message.messageId is not null end" +  // always true
			" and case when :isIncludingTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags))" +
			" else message.messageId is not null end" +  // always true
			" and case when :isExcludingTags then message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
			" else message.messageId is not null end" +  // always true
			" and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
			" else message.messageId is not null end" +  // always true
			" and message.syncState != 'synced' " +
			" and message.uniqueId not in " +
			"(" +
			"SELECT amity_paging_id.id" +
			" from amity_paging_id" +
			" where amity_paging_id.hash = (:hash)" +
			" and amity_paging_id.nonce = (:nonce) " +
			")" +
			" and message.type in (:type)" +
			" order by message.updatedAt desc" +
			" limit 1")
	abstract fun getLatestUnsyncMessageWithTypeImpl(subChannelId: String,
	                                                isFilterByParentId: Boolean?,
	                                                parentId: String?,
	                                                isIncludingTags: Boolean,
	                                                includingTags: Array<String>,
	                                                isExcludingTags: Boolean,
	                                                excludingTags: Array<String>,
	                                                isDeleted: Boolean?,
	                                                type: List<String>,
	                                                hash: Int,
	                                                nonce: Int): Flowable<EkoMessageEntity>
	
	fun getLatestUnsyncMessage(subChannelId: String,
	                           isFilterByParentId: Boolean?,
	                           parentId: String?,
	                           includingTags: Array<String>,
	                           excludingTags: Array<String>,
	                           isDeleted: Boolean?,
	                           type: List<String>,
	                           hash: Int,
	                           nonce: Int): Flowable<EkoMessageEntity> {
		return if (type.isEmpty()) {
			getLatestUnsyncMessageImpl(subChannelId,
					isFilterByParentId,
					parentId,
					includingTags.size > 0,
					includingTags,
					excludingTags.size > 0,
					excludingTags,
					isDeleted,
					hash,
					nonce)
		} else {
			getLatestUnsyncMessageWithTypeImpl(subChannelId,
					isFilterByParentId,
					parentId,
					includingTags.size > 0,
					includingTags,
					excludingTags.size > 0,
					excludingTags,
					isDeleted,
					type,
					hash,
					nonce)
		}
	}
	
	@Query("UPDATE message set messageMarkerHash = :hash where messageId = :messageId")
	abstract fun updateMarkerHash(messageId: String, hash: Int)
	@Query("SELECT * FROM message WHERE createdAt < (SELECT createdAt FROM message WHERE messageId = :messageId) ORDER BY createdAt DESC LIMIT 20")
	abstract fun getCacheBeforeMessageIdImpl(messageId: String): List<EkoMessageEntity>
	fun getCacheBeforeMessageId(messageId: String): List<EkoMessageEntity> {
		return getCacheBeforeMessageIdImpl(messageId)
	}
	
	@Query("SELECT * FROM message WHERE createdAt > (SELECT createdAt FROM message WHERE messageId = :messageId) ORDER BY createdAt ASC LIMIT 20")
	abstract fun getCacheAfterMessageIdImpl(messageId: String): List<EkoMessageEntity>
	fun getCacheAfterMessageId(messageId: String): List<EkoMessageEntity> {
		return getCacheAfterMessageIdImpl(messageId)
	}
}