package com.ekoapp.ekosdk.internal.mapper

import com.amity.socialcloud.sdk.model.chat.notification.AmityChannelNotificationSettings
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.notification.AmityRolesFilter
import com.amity.socialcloud.sdk.model.core.notification.AmityUserNotificationModule
import com.amity.socialcloud.sdk.model.core.notification.AmityUserNotificationModuleName
import com.amity.socialcloud.sdk.model.core.notification.AmityUserNotificationSettings
import com.amity.socialcloud.sdk.model.core.role.AmityRoles
import com.amity.socialcloud.sdk.model.social.notification.AmityCommunityNotificationEvent
import com.amity.socialcloud.sdk.model.social.notification.AmityCommunityNotificationEventName
import com.amity.socialcloud.sdk.model.social.notification.AmityCommunityNotificationSettings
import com.ekoapp.ekosdk.internal.api.dto.EkoNotifiableEventDto
import com.ekoapp.ekosdk.internal.api.dto.EkoNotifiableModuleDto
import com.ekoapp.ekosdk.internal.api.dto.EkoPushNotificationEventDto
import com.ekoapp.ekosdk.internal.api.dto.EkoPushNotificationModuleDto

internal class EkoPushNotificationMapper {

    private fun mapCommunityEvents(events: List<EkoNotifiableEventDto>): List<AmityCommunityNotificationEvent> {
        return events.filter {
            AmityCommunityNotificationEventName.enumOf(it.name) != AmityCommunityNotificationEventName.UNKNOWN_EVENT
        }.map { event ->
            when (AmityCommunityNotificationEventName.enumOf(event.name)) {
                AmityCommunityNotificationEventName.POST_CREATED -> {
                    AmityCommunityNotificationEvent.POST_CREATED(
                            isEnabled = event.isPushNotifiable ?: false,
                            isNetworkEnabled = event.isNetworkEnabled ?: false,
                            rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityCommunityNotificationEventName.POST_REACTED -> {
                    AmityCommunityNotificationEvent.POST_REACTED(
                            isEnabled = event.isPushNotifiable ?: false,
                            isNetworkEnabled = event.isNetworkEnabled ?: false,
                            rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityCommunityNotificationEventName.COMMENT_CREATED -> {
                    AmityCommunityNotificationEvent.COMMENT_CREATED(
                            isEnabled = event.isPushNotifiable ?: false,
                            isNetworkEnabled = event.isNetworkEnabled ?: false,
                            rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityCommunityNotificationEventName.COMMENT_REACTED -> {
                    AmityCommunityNotificationEvent.COMMENT_REACTED(
                            isEnabled = event.isPushNotifiable ?: false,
                            isNetworkEnabled = event.isNetworkEnabled ?: false,
                            rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityCommunityNotificationEventName.COMMENT_REPLIED -> {
                    AmityCommunityNotificationEvent.COMMENT_REPLIED(
                            isEnabled = event.isPushNotifiable ?: false,
                            isNetworkEnabled = event.isNetworkEnabled ?: false,
                            rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityCommunityNotificationEventName.STORY_CREATED -> {
                    AmityCommunityNotificationEvent.STORY_CREATED(
                        isEnabled = event.isPushNotifiable ?: false,
                        isNetworkEnabled = event.isNetworkEnabled ?: false,
                        rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityCommunityNotificationEventName.STORY_REACTED -> {
                    AmityCommunityNotificationEvent.STORY_REACTED(
                        isEnabled = event.isPushNotifiable ?: false,
                        isNetworkEnabled = event.isNetworkEnabled ?: false,
                        rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityCommunityNotificationEventName.STORY_COMMENT_CREATED -> {
                    AmityCommunityNotificationEvent.STORY_COMMENT_CREATED(
                        isEnabled = event.isPushNotifiable ?: false,
                        isNetworkEnabled = event.isNetworkEnabled ?: false,
                        rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                else -> {
                    // should never happen as unknown event is filtered out
                    val exception = AmityException.create("Unrecognized notification module", null, AmityError.MALFORMED_DATA)
                    throw exception
                }
            }
        }
    }

    fun mapToCommunityNotificationSettings(pushNotificationEventDto: EkoPushNotificationEventDto): AmityCommunityNotificationSettings {
        return AmityCommunityNotificationSettings(
                isEnabled = pushNotificationEventDto.isPushNotifiable ?: false,
                events = if (pushNotificationEventDto.notifiableEvents == null) emptyList() else mapCommunityEvents(pushNotificationEventDto.notifiableEvents)
        )
    }

    fun mapToChannelNotificationSettings(pushNotificationEventDto: EkoPushNotificationEventDto): AmityChannelNotificationSettings {
        return AmityChannelNotificationSettings(
                isEnabled = pushNotificationEventDto.isPushNotifiable ?: false
        )
    }

    private fun mapUserEvents(events: List<EkoNotifiableModuleDto>): List<AmityUserNotificationModule> {

        return events.filter {
            AmityUserNotificationModuleName.enumOf(it.moduleName) != AmityUserNotificationModuleName.UNKNOWN_MODULE
        }.map { event ->
            when (AmityUserNotificationModuleName.enumOf(event.moduleName)) {
                AmityUserNotificationModuleName.CHAT -> {
                    AmityUserNotificationModule.CHAT(
                            isEnabled = event.isPushNotifiable ?: false,
                            rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityUserNotificationModuleName.SOCIAL -> {
                    AmityUserNotificationModule.SOCIAL(
                            isEnabled = event.isPushNotifiable ?: false,
                            rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                AmityUserNotificationModuleName.VIDEO_STREAMING -> {
                    AmityUserNotificationModule.VIDEO_STREAMING(
                            isEnabled = event.isPushNotifiable ?: false,
                            rolesFilter = mapRolesFilter(event.listenFromRoleIds, event.ignoreFromRoleIds)
                    )
                }
                else -> {
                    val exception = AmityException.create("Unrecognized notification module", null, AmityError.MALFORMED_DATA)
                    throw exception
                }
            }
        }
    }

    private fun mapRolesFilter(includedRoles: List<String>? = null, excludedRoles: List<String>? = null): AmityRolesFilter {
        if (includedRoles.isNullOrEmpty() && excludedRoles.isNullOrEmpty()) {
            return AmityRolesFilter.All
        } else if (!includedRoles.isNullOrEmpty()) {
            return AmityRolesFilter.ONLY(AmityRoles(includedRoles!!))
        } else {
            return AmityRolesFilter.NOT(AmityRoles(excludedRoles!!))
        }
    }

    fun mapToUserNotificationSettings(pushNotificationModuleDto: EkoPushNotificationModuleDto): AmityUserNotificationSettings {
        return AmityUserNotificationSettings(
                isEnabled = pushNotificationModuleDto.isPushNotifiable ?: false,
                modules = if (pushNotificationModuleDto.notifiableEvents == null) emptyList() else mapUserEvents(pushNotificationModuleDto.notifiableEvents)
        )
    }
}
