package com.amity.socialcloud.sdk.model.core.events

import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.core.domain.topic.*
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.model.chat.message.AmityMessage
import com.amity.socialcloud.sdk.model.chat.subchannel.AmitySubChannel
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import io.reactivex.rxjava3.core.Single

sealed class AmityTopic(
    internal val nonce: String,
    private val id: String,
    private val events: String
) {

    private fun getKey(): String {
        return nonce + id + events
    }

    internal abstract fun generateTopic(): Single<String>

    override fun equals(other: Any?): Boolean {
        return other is AmityTopic && other.getKey() == getKey()
    }

    override fun hashCode(): Int {
        return getKey().hashCode()
    }

    class COMMUNITY(val community: AmityCommunity, val events: AmityCommunityEvents) :
        AmityTopic("community", community.getCommunityId(), events.name) {

        override fun generateTopic(): Single<String> {
            return CommunityTopicGeneratorUseCase().execute(community, events)
        }

    }

    class COMMENT(val comment: AmityComment, val events: AmityCommentEvents) :
        AmityTopic("comment", comment.getCommentId(), events.name) {

        override fun generateTopic(): Single<String> {
            return CommentTopicGeneratorUseCase().execute(comment, events)
        }

    }

    class USER(val user: AmityUser, val events: AmityUserEvents) :
        AmityTopic("user", user.getUserId(), events.name) {

        override fun generateTopic(): Single<String> {
            return UserTopicGeneratorUseCase().execute(user, events)
        }

    }

    class POST(val post: AmityPost, val events: AmityPostEvents) :
        AmityTopic("post", post.getPostId(), events.name) {

        override fun generateTopic(): Single<String> {
            return PostTopicGeneratorUseCase().execute(post, events)
        }
    }

    class STORY(val story: AmityStory) :
        AmityTopic("story", story.getStoryId(), "") {

        override fun generateTopic(): Single<String> {
            return StoryTopicGeneratorUseCase().execute(story)
        }
    }

    class FOLLOW(val events: AmityFollowEvents) :
        AmityTopic("follow", AmityCoreClient.getUserId(), events.name) {

        override fun generateTopic(): Single<String> {
            return FollowTopicGeneratorUseCase().execute(events)
        }
    }

    internal class CHANNEL(val channel: AmityChannel) :
        AmityTopic("channel", channel.getChannelId(), "") {

        override fun generateTopic(): Single<String> {
            return ChannelTopicGeneratorUseCase().execute(channel)
        }
    }

    internal class SUB_CHANNEL(val subChannel: AmitySubChannel) :
        AmityTopic("sub_channel", subChannel.getSubChannelId(), "") {

        override fun generateTopic(): Single<String> {
            return SubChannelTopicGeneratorUseCase().execute(subChannel)
        }
    }

    internal class MESSAGE(val message: AmityMessage) :
        AmityTopic("message", message.getMessageId(), "") {

        override fun generateTopic(): Single<String> {
            return MessageTopicGeneratorUseCase().execute(message)
        }
    }


    internal class NETWORK
        : AmityTopic("network", AmityCoreClient.getUserId(), "") {

        override fun generateTopic(): Single<String> {
            return NetworkTopicGeneratorUseCase().execute()
        }
    }

    internal class SMART_CHANNEL
        : AmityTopic("smart_channel", AmityCoreClient.getUserId(), "") {

        override fun generateTopic(): Single<String> {
            return SmartChannelTopicGeneratorUseCase().execute()
        }
    }

    internal class SMART_MESSAGE
        : AmityTopic("smart_message", AmityCoreClient.getUserId(), "") {

        override fun generateTopic(): Single<String> {
            return SmartMessageTopicGeneratorUseCase().execute()
        }
    }

    internal class SMART_MESSAGE_FEED
        : AmityTopic("smart_message_feed", AmityCoreClient.getUserId(), "") {

        override fun generateTopic(): Single<String> {
            return SmartMessageFeedTopicGeneratorUseCase().execute()
        }
    }
    
    internal class SUB_CHANNEL_MARKER(val subChannelId: String, val channelId: String)
        : AmityTopic("sub_channel_marker", AmityCoreClient.getUserId(), "") {
        
        override fun generateTopic(): Single<String> {
            return SubChannelMarkerTopicGeneratorUseCase().execute(subChannelId, channelId)
        }
    }
    
    internal class USER_MARKER
        : AmityTopic("user_marker", AmityCoreClient.getUserId(), "") {
        
        override fun generateTopic(): Single<String> {
            return UserMarkerTopicGeneratorUseCase().execute()
        }
    }
}