package com.amity.socialcloud.sdk.api.core.user

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.api.core.ExperimentalAmityApi
import com.amity.socialcloud.sdk.api.core.follow.AmityRelationShip
import com.amity.socialcloud.sdk.api.core.user.flag.AmityUserFlagger
import com.amity.socialcloud.sdk.api.core.user.query.AmityUserQuery
import com.amity.socialcloud.sdk.api.core.user.search.AmityUserSearch
import com.amity.socialcloud.sdk.core.domain.user.IsUserFlaggedByMeUseCase
import com.amity.socialcloud.sdk.core.domain.user.UserGetByIdsUseCase
import com.amity.socialcloud.sdk.model.core.analytics.AmityViewedType
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.ekoapp.ekosdk.internal.usecase.user.FlagUserUseCase
import com.ekoapp.ekosdk.internal.usecase.user.GetBlockedUsersPagingDataUseCase
import com.ekoapp.ekosdk.internal.usecase.user.GetViewedUsersPagingDataUseCase
import com.ekoapp.ekosdk.internal.usecase.user.UnFlagUserUseCase
import com.ekoapp.ekosdk.internal.usecase.user.UserLiveObjectUseCase
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

class AmityUserRepository internal constructor() {

    /* begin_public_function
	  id: user.query
	  api_style: live_collection
	*/
    fun getUsers(): AmityUserQuery.Builder {
        return AmityUserQuery.Builder()
    }
    /* end_public_function */

    /* begin_public_function
	  id: user.search
	  api_style: live_collection
	*/
    fun searchUsers(keyword: String): AmityUserSearch.Builder {
        return AmityUserSearch.Builder().keyword(keyword)
    }
    /* end_public_function */

    @Deprecated("Use searchUsers(keyword: String) instead")
    fun searchUserByDisplayName(keyword: String): AmityUserSearch.Builder {
        return searchUsers(keyword)
    }

    /* begin_public_function
	  id: user.get
	*/
    fun getUser(userId: String): Flowable<AmityUser> {
        return UserLiveObjectUseCase().execute(userId)
    }
    /* end_public_function */

    fun getCurrentUser(): Flowable<AmityUser> {
        return getUser(AmityCoreClient.getUserId())
    }

    /* begin_public_function
	  id: user.get_by_ids
	  api_style: live_collection
	*/
    fun getUserByIds(userIds: Set<String>): Flowable<List<AmityUser>> {
        return UserGetByIdsUseCase().execute(userIds)
    }
    /* end_public_function */

    /* begin_public_function
	  id: user.flag
	*/
    fun flagUser(userId: String): Completable {
        return FlagUserUseCase().execute(userId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: user.unflag
	*/
    fun unflagUser(userId: String): Completable {
        return UnFlagUserUseCase().execute(userId)
    }
    /* end_public_function */

    /* begin_public_function
	  id: user.get_blocked_users
	*/
    fun getBlockedUsers(): Flowable<PagingData<AmityUser>> {
        return GetBlockedUsersPagingDataUseCase().execute()
    }
    /* end_public_function */

    @Deprecated("Use AmityCoreClient.newUserRepository() instead")
    fun report(userId: String): AmityUserFlagger {
        return AmityUserFlagger(userId)
    }

    fun relationship(): AmityRelationShip {
        return AmityRelationShip()
    }

    /* begin_public_function
       id: user.check_flag_by_me
       api_style: async
   */
    @ExperimentalAmityApi
    fun isUserFlaggedByMe(userId: String): Single<Boolean> {
        return Single.fromCallable { IsUserFlaggedByMeUseCase().execute(userId) }
    }
    /* end_public_function */


    @Deprecated("Use getReachUsers() instead")
    fun getViewedUsers(postId: String): Flowable<PagingData<AmityUser>> {
        return getReachedUsers(AmityViewedType.POST, postId)
    }

    /* begin_public_function
         id: user.get_viewed_users
         api_style: live_collection
       */
    fun getReachedUsers(viewedType: AmityViewedType, viewedId: String): Flowable<PagingData<AmityUser>> {
        return GetViewedUsersPagingDataUseCase().execute(
            viewableItemType = viewedType,
            viewableItemId = viewedId
        )    }
    /* end_public_function */
}
