package com.ekoapp.ekosdk.internal.repository.post

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.data.post.PostQueryPersister
import com.amity.socialcloud.sdk.social.data.post.PostRemoteDataStore
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.api.dto.EkoPostQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.keycreator.UserPostKeyCreator
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.usecase.post.PostReactorUseCase
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@ExperimentalPagingApi
internal class UserPostMediator(
    private val userId: String,
    private val sortBy: String,
    private val isDeleted: Boolean?,
    private val postTypes: List<AmityPost.DataType>,
) : DynamicQueryStreamMediator<PostEntity, EkoPostQueryDto, AmityPost>(
    nonce = AmityNonce.USER_FEED,
    dynamicQueryStreamKeyCreator = UserPostKeyCreator(
        userId = userId,
        sortBy = sortBy,
        isDeleted = isDeleted,
        postTypes = postTypes
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityPost> {
        return PostReactorUseCase().execute(
            targetType = AmityPost.TargetType.USER,
            targetId = userId,
            includeDeleted = isDeleted,
            postTypes = postTypes,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = AmityNonce.USER_FEED
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EkoPostQueryDto> {
        return getRequest(limit = pageSize)

    }

    override fun getFetchByTokenRequest(token: String): Single<EkoPostQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EkoPostQueryDto): Completable {
        return PostQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EkoPostQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.paging.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.posts?.map { it.postId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EkoPostQueryDto> {
        val matchingOnlyParentPost = getDataTypes()?.isEmpty() ?: true
        return PostRemoteDataStore().getUserPosts(
            userId = userId,
            isDeleted = isDeleted,
            postTypes = getDataTypes(),
            sortBy = sortBy,
            matchingOnlyParentPost = matchingOnlyParentPost,
            limit = limit,
            token = token
        ).toRx2()
    }

    private fun getDataTypes(): List<String>? {
        if (postTypes.isNotEmpty()) {
            val typeArrayList = arrayListOf<String>()
            postTypes.forEach { typeArrayList.add(it.getApiKey()) }
            return typeArrayList
        }
        return null
    }
}