package com.amity.socialcloud.sdk.social.data.community.membership

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.social.member.query.AmityCommunityMembershipSortOption
import com.amity.socialcloud.sdk.model.core.role.AmityRoles
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMember
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMembership
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMembershipFilter
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.membership.paging.CommunityMemberDynamicQueryMediator
import com.amity.socialcloud.sdk.social.data.community.membership.paging.CommunityMemberQueryMediator
import com.amity.socialcloud.sdk.social.data.community.membership.paging.CommunityMemberSearchMediator
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable

@OptIn(ExperimentalPagingApi::class)
internal class CommunityMembershipRepository : EkoObjectRepository() {

    fun addUsers(communityId: String, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().addUsers(communityId, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun removeUsers(communityId: String, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().removeUsers(communityId, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun addRoles(communityId: String, roles: List<String>, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().addRoles(communityId, roles, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun removeRoles(communityId: String, roles: List<String>, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().removeRoles(communityId, roles, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun banUsers(communityId: String, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().banUsers(communityId, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun unbanUsers(communityId: String, userIds: List<String>): Completable {
        return CommunityMembershipRemoteDataStore().unbanUsers(communityId, userIds)
            .flatMapCompletable {
                CommunityQueryPersister().persist(it)
            }
    }

    fun getMembership(communityId: String, userId: String): AmityCommunityMember? {
        val member = CommunityMembershipLocalDataStore().getMembership(communityId, userId)
        return if (member == null) null else CommunityMembershipModelMapper().map(member)
    }

    fun searchCommunityMembershipPagingData(
        communityId: String,
        roles: AmityRoles,
        communityMembership: List<AmityCommunityMembership>?,
        sortBy: AmityCommunityMembershipSortOption,
        keyword: String
    ): Flowable<PagingData<AmityCommunityMember>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = DEFAULT_PAGE_SIZE,
                enablePlaceholders = false
            ),
            queryStreamMediator = CommunityMemberSearchMediator(
                communityId = communityId,
                roles = roles,
                filters = communityMembership,
                sortBy = sortBy,
                keyword = keyword,
            ),
            pagingSourceFactory = {
                CommunityMembershipLocalDataStore().getMemberSearchPagingSource(
                    communityId,
                    roles,
                    communityMembership,
                    sortBy,
                    keyword
                )
            },
            modelMapper = CommunityMembershipModelMapper()
        )

        return pagerCreator.create().toRx3()
    }

    fun getCommunityMembershipPagingData(
        communityId: String,
        roles: AmityRoles,
        filters: AmityCommunityMembershipFilter,
        sortBy: AmityCommunityMembershipSortOption
    ): Flowable<PagingData<AmityCommunityMember>> {
        val memberships = filters.memberships.map(AmityCommunityMembership::enumOf)
        val pagerCreator = if (sortBy == AmityCommunityMembershipSortOption.DISPLAY_NAME) {
            QueryStreamPagerCreator(
                pagingConfig = PagingConfig(
                    pageSize = getDefaultPageSize(),
                    enablePlaceholders = false
                ),
                queryStreamMediator = CommunityMemberQueryMediator(
                    communityId,
                    roles,
                    memberships,
                    sortBy
                ),
                pagingSourceFactory = {
                    CommunityMembershipLocalDataStore().getMemberQueryPagingSource(
                        communityId,
                        roles,
                        memberships,
                        sortBy
                    )
                },
                modelMapper = CommunityMembershipModelMapper()
            )
        } else {
            DynamicQueryStreamPagerCreator(
                pagingConfig = PagingConfig(
                    pageSize = getDefaultPageSize(),
                    enablePlaceholders = false
                ),
                dynamicQueryStreamMediator = CommunityMemberDynamicQueryMediator(
                    communityId,
                    roles,
                    memberships,
                    sortBy
                ),
                pagingSourceFactory = {
                    CommunityMembershipLocalDataStore().getMemberDynamicQueryPagingSource(
                        communityId,
                        roles,
                        memberships,
                        sortBy
                    )
                },
                modelMapper = CommunityMembershipModelMapper()
            )
        }
        return pagerCreator.create().toRx3()
    }

    fun getLatestCommunityMember(
        communityId: String,
        roles: AmityRoles,
        memberships: List<AmityCommunityMembership>,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<AmityCommunityMember> {
        return CommunityMembershipLocalDataStore().getLatestCommunityMember(
            communityId = communityId,
            memberships = memberships,
            roles = roles,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        )
            .map {
                CommunityMembershipModelMapper().map(it)
            }
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }
}