package com.amity.socialcloud.sdk.social.data.community.membership.paging

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.api.social.member.query.AmityCommunityMembershipSortOption
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.core.role.AmityRoles
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMember
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMembership
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.membership.CommunityMembershipRemoteDataStore
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.CommunityMembershipEntity
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

class CommunityMemberQueryMediator(
    private val communityId: String,
    private val roles: AmityRoles,
    private val filters: List<AmityCommunityMembership>,
    private val sortBy: AmityCommunityMembershipSortOption
) : DynamicQueryStreamMediator<CommunityMembershipEntity, CommunityListQueryDto, AmityCommunityMember>(
    nonce = AmityNonce.COMMUNITY_MEMBERSHIP,
    dynamicQueryStreamKeyCreator = CommunityMemberQueryKeyCreator(
        communityId,
        roles,
        filters,
        sortBy,
    ),
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityCommunityMember> {
        return Flowable.empty()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<CommunityListQueryDto> {
        val options = QueryOptionsRequestParams(limit = pageSize)
        return CommunityMembershipRemoteDataStore().queryCommunityUsers(
            communityId = communityId,
            memberships = filters.map { it.value },
            roles = roles,
            sortBy = sortBy.apiKey,
            options = options
        ).toRx2()
    }

    override fun getFetchByTokenRequest(token: String): Single<CommunityListQueryDto> {
        val options = QueryOptionsRequestParams(token = token)
        return CommunityMembershipRemoteDataStore().queryCommunityUsers(
            communityId = communityId,
            memberships = filters.map { it.value },
            roles = roles,
            sortBy = sortBy.apiKey,
            options = options
        ).toRx2()
    }

    override fun persistResponse(dto: CommunityListQueryDto): Completable {
        return CommunityQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: CommunityListQueryDto): QueryStreamQueryToken {
        return QueryStreamQueryToken(
            queryStreamKeyCreator = CommunityMemberQueryKeyCreator(
                communityId,
                roles,
                filters,
                sortBy,
            ),
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.communityUsers.map { user -> user.userId ?: "" }
        )
    }

}