package com.amity.socialcloud.sdk.social.data.community.membership

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.api.social.member.query.AmityCommunityMembershipSortOption
import com.amity.socialcloud.sdk.model.chat.member.AmityMembershipType
import com.amity.socialcloud.sdk.model.core.permission.AmityPermissions
import com.amity.socialcloud.sdk.model.core.role.AmityRoles
import com.amity.socialcloud.sdk.model.social.member.AmityCommunityMembership
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.entity.CommunityMembershipEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class CommunityMembershipLocalDataStore {

    // EkoMyCommunityMembershipPersister
    fun saveCommunities(communities: List<CommunityEntity>): Completable {
        return Completable.fromAction {
            communities.forEach {
                val dao = UserDatabase.get().communityMembershipDao()
                if (dao.getByCommunityIdAndUserIdNow(
                        it.communityId,
                        AmityCoreClient.getUserId()
                    ) == null
                ) {
                    val defaultMembership = CommunityMembershipEntity()
                    defaultMembership.communityId = it.communityId
                    defaultMembership.userId = AmityCoreClient.getUserId()
                    defaultMembership.roles = AmityRoles()
                    defaultMembership.permissions = AmityPermissions()
                    defaultMembership.communityMembership = when (it.isJoined ?: false) {
                        true -> AmityMembershipType.MEMBER.apiKey
                        false -> AmityMembershipType.NONE.apiKey
                    }
                    defaultMembership.createdAt = DateTime.now()
                    defaultMembership.updatedAt = DateTime.now()
                    dao.insert(`object` = defaultMembership)
                }
            }
        }
    }

    // EkoCommunityMembershipListPersister
    fun saveMemberships(memberships: List<CommunityMembershipEntity>): Completable {
        return Completable.fromAction {
            UserDatabase.get().communityMembershipDao().save(memberships)
        }
    }

    fun getMembership(communityId: String, userId: String): CommunityMembershipEntity? {
        var membership: CommunityMembershipEntity? = null
        Completable.fromCallable {
            val membershipDao = UserDatabase.get().communityMembershipDao()
            val entity = membershipDao.getByCommunityIdAndUserIdNow(communityId, userId)
            if (entity != null) {
                membership = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return membership
    }

    fun getMemberQueryPagingSource(
        communityId: String,
        roles: AmityRoles,
        filters: List<AmityCommunityMembership>,
        sortBy: AmityCommunityMembershipSortOption,
    ): PagingSource<Int, CommunityMembershipEntity> {
        return UserDatabase.get().communityMembersPagingDao()
            .queryCommunityMembership(communityId, roles, filters, sortBy)
    }

    fun getMemberDynamicQueryPagingSource(
        communityId: String,
        roles: AmityRoles,
        filters: List<AmityCommunityMembership>,
        sortBy: AmityCommunityMembershipSortOption,
    ): PagingSource<Int, CommunityMembershipEntity> {
        return UserDatabase.get().communityMembersDynamicQueryPagingDao()
            .queryCommunityMembership(communityId, roles, filters, sortBy)
    }

    fun getMemberSearchPagingSource(
        communityId: String,
        roles: AmityRoles,
        filters: List<AmityCommunityMembership>?,
        sortBy: AmityCommunityMembershipSortOption,
        keyword: String? = null
    ): PagingSource<Int, CommunityMembershipEntity> {
        return UserDatabase.get().communityMemberSearchPagingDao()
            .searchCommunityMembership(communityId, roles, filters, sortBy, keyword)
    }

    fun getLatestCommunityMember(
        communityId: String,
        memberships: List<AmityCommunityMembership>,
        roles: AmityRoles,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<CommunityMembershipEntity> {
        return UserDatabase.get().communityMembershipDao().getLatestCommunityMembership(
            communityId,
            memberships.map { it.value },
            roles,
            dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce,
            DateTime.now()
        )
    }

    fun notifyUserUpdate(userId: String) {
        UserDatabase.get().communityMembershipDao().updateUser(userId)
    }

}