package com.amity.socialcloud.sdk.log

import android.util.Log
import java.io.PrintWriter
import java.io.StringWriter

object AmityLog {
	
	private val explicitTag = ThreadLocal<String>()
	
	private fun getTag(): String? {
		val tag = explicitTag.get()
		if (tag != null) {
			explicitTag.remove()
		}
		return tag
	}

	fun tag(tag: String): AmityLog {
		explicitTag.set(tag)
		return this
	}
	
	fun v(message: String, vararg args: Any?) {
		v(t = null, message = message, args = *args)
	}
	
	/** Log a verbose exception and a message with optional format args.  */
	fun v(t: Throwable?, message: String?, vararg args: Any?) {
		prepareLog(Log.VERBOSE, t, message, *args)
	}
	
	/** Log a verbose exception.  */
	fun v(t: Throwable?) {
		v(t = t, message = null)
	}
	
	/** Log a debug message with optional format args.  */
	fun d(message: String?, vararg args: Any?) {
		d(t = null, message = message, args = *args)
	}
	
	/** Log a debug exception and a message with optional format args.  */
	fun d(t: Throwable?, message: String?, vararg args: Any?) {
		prepareLog(Log.DEBUG, t, message, *args)
	}
	
	/** Log a debug exception.  */
	fun d(t: Throwable?) {
		d(t = t, message = null)
	}
	
	/** Log an info message with optional format args.  */
	fun i(message: String?, vararg args: Any?) {
		i(t = null, message = message, args = *args)
	}
	
	/** Log an info exception and a message with optional format args.  */
	fun i(t: Throwable?, message: String?, vararg args: Any?) {
		prepareLog(Log.INFO, t, message, *args)
	}
	
	/** Log an info exception.  */
	fun i(t: Throwable?) {
		i(t = t, message = null)
	}
	
	/** Log a warning message with optional format args.  */
	fun w(message: String?, vararg args: Any?) {
		w(t = null, message = message, args = *args)
	}
	
	/** Log a warning exception and a message with optional format args.  */
	fun w(t: Throwable?, message: String?, vararg args: Any?) {
		prepareLog(Log.WARN, t, message, *args)
	}
	
	/** Log a warning exception.  */
	fun w(t: Throwable?) {
		w(t = t, message = null)
	}
	
	/** Log an error message with optional format args.  */
	fun e(message: String?, vararg args: Any?) {
		e(t = null, message = message, args = *args)
	}
	
	/** Log an error exception and a message with optional format args.  */
	fun e(t: Throwable?, message: String?, vararg args: Any?) {
		prepareLog(Log.ERROR, t, message, *args)
	}
	
	/** Log an error exception.  */
	fun e(t: Throwable?) {
		e(t = t, message = null)
	}
	
	private fun prepareLog(priority: Int, t: Throwable?, message: String?, vararg args: Any?) {
		// Consume tag even when message is not loggable so that next message is correctly tagged.
		val tag = getTag()
		
		var message: String = message ?: ""
		
		if (!isLoggable()) {
			return
		}
		if (message.isEmpty()) {
			if (t == null) {
				return  // Swallow message if it's null and there's no throwable.
			}
			message = getStackTraceString(t)
		} else {
			if (args.isNotEmpty()) {
				message = String.format(message, *args)
			}
			if (t != null) {
				message += """
	        	
	        	${getStackTraceString(t)}
	        	""".trimIndent()
			}
		}
		log(priority, tag, message, t)
	}
	
	private fun log(priority: Int, tag: String?, message: String, t: Throwable?) {
		when(priority){
			Log.VERBOSE -> {
				Log.v(tag,message,t)
			}
			Log.INFO -> {
				Log.i(tag,message,t)
			}
			Log.DEBUG -> {
				Log.d(tag,message,t)
			}
			Log.WARN -> {
				Log.w(tag,message,t)
			}
			Log.ERROR -> {
				Log.e(tag,message,t)
			}
		}
	}
	
	private fun isLoggable(): Boolean {
		return !AmityLogProvider.isHiddenAmityLog()
	}
	
	private fun getStackTraceString(t: Throwable): String {
		// Don't replace this with Log.getStackTraceString() - it hides
		// UnknownHostException, which is not what we want.
		val sw = StringWriter(256)
		val pw = PrintWriter(sw, false)
		t.printStackTrace(pw)
		pw.flush()
		return sw.toString()
	}
}