package com.flybits.android.push.deserializations

import com.flybits.android.push.models.PushToken
import com.flybits.commons.library.deserializations.IDeserializer

import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject


const val DEVICE_TOKEN = "deviceToken"
const val NETWORK = "network"
const val PUSH_VERSION = "pushVersion"
const val DATA = "data"

/**
 * Deserializes and serializes the [PushToken] from and to JSON.
 */
class DeserializePushToken : IDeserializer<PushToken>() {

    /**
     * Serialize the [PushToken] to JSON [String].
     *
     * @param obj [PushToken] to be serialized.
     *
     * @return serialized [PushToken]
     */
    override fun toJson(obj: PushToken?): String {

        if (obj == null) {
            return "{}"
        }

        val jsonObject = JSONObject()

        try {
            jsonObject.put(DEVICE_TOKEN, obj.deviceToken)
            jsonObject.put(NETWORK, obj.type)
            jsonObject.put(PUSH_VERSION, obj.pushVersion)
            if (obj.data.isNotEmpty()) {
                val jsonObjectData = JSONObject(obj.data)
                jsonObject.put(DATA, jsonObjectData)
            }
        } catch (exception: JSONException) {
        }

        return jsonObject.toString()
    }

    /**
     * Deserialize the JSON String to [PushToken] object.
     *
     * @param jObj String containing the JSON [PushToken] payload.
     *
     * @return Deserialized [PushToken]
     */
    override fun fromJson(jObj: JSONObject): PushToken? {
        return try {
            val dataMap = if (jObj.has(DATA) && !jObj.isNull(DATA)) {
                val dataObj = jObj.getJSONObject(DATA)
                dataObj.keys().asSequence().map {
                    Pair(it.toString(), dataObj[it].toString())
                }.toMap()
            } else emptyMap()

            val pushVersion = if (jObj.has(PUSH_VERSION)) {
                jObj.getInt(PUSH_VERSION)
            } else 1

            PushToken(deviceToken = jObj.getString(DEVICE_TOKEN)
                , pushVersion = pushVersion
                , data = dataMap)
        } catch (exceptions: JSONException) {
            null
        }

    }

    override fun fromJson(jsonArray: JSONArray): PushToken? {
        return null
    }
}
