package com.flybits.android.push.db.caching;

import android.app.Activity;
import android.content.Context;

import com.flybits.android.push.db.PushDatabase;
import com.flybits.android.push.models.Push;
import com.flybits.commons.library.caching.FlybitsCacheLoader;

import java.util.List;

/**
 * The {@code PushCacheLoader} is responsible for loading cached data associated to a {@link Push}.
 * This data is inserted into the local SQLite DB whenever {@link Push} data is received from the
 * Flybits server.
 */
public class PushCacheLoader extends FlybitsCacheLoader<Push> {

    /**
     * The Push Key that is used internally for storing the cached push notifications.
     */
    public static final String PUSH_CACHE_KEY   = "com.flybits.caching.push.NOTIFICATIONS";

    /**
     * Constructor used to define a {@link PushCacheLoader} class with the Activity parameter.
     *
     * @param activity The activity that created this object.
     */
    public PushCacheLoader(Activity activity){
        super(activity);
    }

    @Override
    protected List<Push> load(List<String> cachedIds, int limit) {
        return PushDatabase.getDatabase(getContext()).pushDao().getByCachedIds(cachedIds, limit);
    }

    @Override
    protected Push load(String id) {
        if (id != null) {
            return PushDatabase.getDatabase(getContext()).pushDao().getById(id);
        }
        return null;
    }

    /**
     * This get method is used to retrieve the cached entities associated to the {@link Push}
     * model.
     *
     * @param callback The {@code CacheListLoader} callback that loads the cached data.
     * @return This {@link #PushCacheLoader} object.
     */
    public PushCacheLoader get(final CacheListLoader<Push> callback){
        super.getList(PUSH_CACHE_KEY, callback);
        return this;
    }

    /**
     * This get method is used to retrieve the cached entities associated to the {@link Push}
     * model.
     *
     * @param limit The number of records that should be returned.
     * @param callback The {@code CacheListLoader} callback that loads the cached data.
     * @return This {@link #PushCacheLoader} object.
     */
    public PushCacheLoader get(int limit, CacheListLoader<Push> callback){
        super.getList(PUSH_CACHE_KEY, limit, callback);
        return this;
    }
}
