package com.flybits.android.kernel.models;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;

import com.flybits.android.kernel.api.FlyContentData;
import com.flybits.android.kernel.models.internal.ContentDataResponse;
import com.flybits.android.kernel.utilities.ContentDataDeserializer;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.Result;

import java.util.ArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * A object representing an array of data in a Content Data response. Used for arrays when defining
 * your POJO. After initial load, more data can be requested using {@code getNext}.
 * @param <T> The class type this array holds.
 */
public class PagedArray<T extends Parcelable> implements Parcelable{

    private String mContentInstanceId;
    private String mFieldPath;
    private long mCurrentOffset = 0;
    private long mTotalRecords;
    private long mLimit;
    private ArrayList<T> mCurrentDownloaded;
    private Class mTemplateModel;

    /**
     * The constructor used to define all the necessary variables to define a paged list of
     * {@link Content}.
     *
     * @param contentInstanceId The unique identifier that represents a specific {@link Content}
     *                          instance that this list of data is associated to. This unique
     *                          identifier is in the form of GUID.
     * @param fieldPath The key name that holds the list of {@code ContentData}.
     * @param model The object model that each item in the {@code ContentData} follows.
     * @param totalRecords The total number of items within the list of {@code ContentData}.
     * @param limit The maximum number of {@code ContentData} returned as part of this
     *              {@link Content}.
     * @param offset The item number from which the current list of {@code ContentData} is retrieved
     *               from.
     * @param list The list of {@code ContentData} to be displayed to the end user.
     */
    public PagedArray(String contentInstanceId, String fieldPath, Class model,
                      long totalRecords, long limit, long offset, ArrayList<T> list) {
        mContentInstanceId      = contentInstanceId;
        mFieldPath              = fieldPath;
        mTotalRecords           = totalRecords;
        mLimit                  = limit;
        mCurrentOffset          = offset + limit;

        if (mCurrentOffset >= mTotalRecords) {
            mCurrentOffset = mTotalRecords;
        }
        mCurrentDownloaded      = list;
        mTemplateModel          = model;
    }

    /**
     * Constructor used to unmarshall {@code Parcel} object.
     *
     * @param in The {@code Parcel} that contains information about the {@code PagedArray}.
     */
    public PagedArray(Parcel in) {
        mContentInstanceId      = in.readString();
        mFieldPath              = in.readString();
        mCurrentOffset          = in.readLong();
        mTotalRecords           = in.readLong();
        mLimit                  = in.readLong();

        int listSize            = in.readInt();
        if (listSize > 0){
            Class<?> type       = (Class<?>) in.readSerializable();
            mCurrentDownloaded  = new ArrayList<>(listSize);
            in.readList(mCurrentDownloaded, type.getClassLoader());
        }

        try {
            mTemplateModel      = Class.forName(in.readString());
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
    }

    /**
     * Gets the instance id of the {@code Content} that held this {@code PagedArray}.
     *
     * @return The instance identifier of the parent {@code Content}.
     */
    public String getInstanceId() {
        return mContentInstanceId;
    }

    /**
     * The maximum number of items returned through the last retrieval of {@code ContentData}.
     *
     * @return The maximum number of items returned in the last request for items.
     */
    public long getLimit(){
        return mLimit;
    }

    /**
     * Gets a list of currently downloaded items. If no items are currently downloaded then an empty
     * list will be returned.
     *
     * @return The list of downloaded items based on the last request. If no items were retrieved,
     * an empty list will returned
     */
    public ArrayList<T> getList()
    {
        if (mCurrentDownloaded != null) {
            return mCurrentDownloaded;
        }
        return new ArrayList<>();
    }

    /**
     * Gets the name of this {@code PagedArray}.
     *
     * @return The name of this {@code PagedArray}.
     */
    public String getName()
    {
        if (mFieldPath.contains("."))
            return mFieldPath.substring(mFieldPath.lastIndexOf(".") + 1);
        return mFieldPath;
    }

    /**
     * Requests the next page of data from the server.
     *
     * @param context The current context of the application.
     * @param limit The max amount of items to download for the next page.
     *
     * @throws FlybitsException in the event that something went wrong while trying to retrieve the
     * next list of {@code ContentData} or while attempting to parse it.
     */
    public void getNext(Context context, int limit) throws FlybitsException {
        if (!hasNext())
        return;

        Result<ContentDataResponse<T>> result = FlyContentData.getPaged(context, PagedArray.this, mCurrentOffset, limit);
        PagedArray<T> newPagedArray = ContentDataDeserializer.extractPagedArray(result.getResult().getItems().get(0), mFieldPath);

        if (newPagedArray != null) {
            mTotalRecords = newPagedArray.mTotalRecords;

            mCurrentOffset += limit;
            mLimit = newPagedArray.mLimit;
            mCurrentDownloaded.addAll(newPagedArray.getList());

            if (mCurrentOffset >= mTotalRecords)
                mCurrentOffset = mTotalRecords;
        }
    }

    /**
     * Asynchronously get a list of {@code Content} based on the
     * {@link com.flybits.android.kernel.utilities.ContentParameters} parameter which defines
     * what options should be used to filter the list of returned Content.
     *
     * @param context The context of the application.
     * @param limit The max amount of items to download for the next page.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     * {@code Experiences} based on the {@code params}.
     */
    public void getNext(final Context context, final long limit, final BasicResultCallback callback){

        if (!hasNext()){
            callback.onSuccess();
        }

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult result = new BasicResult(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    final Result<ContentDataResponse<T>> pageResult = FlyContentData.getPaged(context, PagedArray.this, mCurrentOffset, limit);
                    PagedArray<T> newPagedArray = ContentDataDeserializer.extractPagedArray(pageResult.getResult().getItems().get(0), mFieldPath);
                    if (newPagedArray == null){
                        result.setFailed(pageResult.getException());
                    }else{
                        mTotalRecords = newPagedArray.mTotalRecords;

                        mCurrentOffset += limit;
                        mLimit = newPagedArray.mLimit;
                        mCurrentDownloaded.addAll(newPagedArray.getList());

                        if (mCurrentOffset >= mTotalRecords)
                            mCurrentOffset = mTotalRecords;

                        result.setResult(pageResult);
                    }

                }catch (final FlybitsException e){
                    result.setFailed(e);
                }
            }
        });
    }

    /**
     * Get the list item position from which the current list of {@code ContentData} is retrieved
     * from.
     *
     * @return The list item position from which the current list of {@code ContentData} is
     * retrieved from.
     */
    public long getNextOffset(){
        return mCurrentOffset;
    }

    /**
     * Gets the field path to this {@code PagedArray}.
     *
     * @return The field path to this {@code PagedArray}.
     */
    public String getPath()
    {
        return mFieldPath;
    }

    /**
     * Get's the model of the template that held this Paged Array. Used for paging and deserialization.
     * @return A class object of the template this PagedArray resides in.
     */
    public Class getTemplateModel() {
        return mTemplateModel;
    }

    /**
     * Get the total number of items presents in this {@code ContentData}.
     *
     * @return the total number of items present in this {@code ContentData}.
     */
    public long getNumTotalRecords(){
        return mTotalRecords;
    }

    /**
     * Returns if there is more items to request from the server.
     *
     * @return True if there is more items from the server.
     */
    public boolean hasNext()
    {
        return mCurrentOffset < mTotalRecords;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(mContentInstanceId);
        dest.writeString(mFieldPath);
        dest.writeLong(mCurrentOffset);
        dest.writeLong(mTotalRecords);
        dest.writeLong(mLimit);
        dest.writeInt(mCurrentDownloaded.size());

        if (mCurrentDownloaded != null && mCurrentDownloaded.size() > 0){
            final Class<?> objectsType = mCurrentDownloaded.get(0).getClass();
            dest.writeSerializable(objectsType);
            dest.writeList(mCurrentDownloaded);
        }

        dest.writeString(mTemplateModel.getName());
    }

    public static final Creator<PagedArray> CREATOR = new Creator<PagedArray>() {
        @Override
        public PagedArray createFromParcel(Parcel in) {
            return new PagedArray(in);
        }

        @Override
        public PagedArray[] newArray(int size) {
            return new PagedArray[size];
        }
    };
}
