package com.flybits.context.workers

import android.content.Context
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.context.ContextManager

/**
 * This is a worker for WorkManager API for scheduling a periodic upload of unsent Context Plugin data
 * to Flybits server after every given interval (minimum is 15 minutes).
 *
 */
class ContextUploadingWorker(var context: Context, workerParameters: WorkerParameters) :
    Worker(context, workerParameters) {
    override fun doWork(): Result {
        return doWork(context)
    }

    companion object {
        private const val TAG_LOGGING = "CtxWorker"
        fun doWork(context: Context): Result {
            return try {
                synchronized(this) {
                    Logger.appendTag(TAG_LOGGING)
                        .i("Starting Context Plugin Fetching & Uploading...")
                    // This will refresh the context plugin if any running and will flush data to server.
                    ContextManager.refreshAll(context, null)
                    Logger.appendTag(TAG_LOGGING).i("Done Context Uploading...")
                    Result.success()
                }
            } catch (e: FlybitsException) {
                Logger.appendTag(TAG_LOGGING).e("Error Context Uploading...", e)
                Result.failure()
            }
        }
    }
}