package com.flybits.context.plugins.location


import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.location.LocationManager
import android.os.Build
import android.os.Bundle
import android.support.v4.content.ContextCompat
import androidx.work.WorkerParameters
import com.flybits.commons.library.SharedElements
import com.flybits.commons.library.SharedElementsFactory
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.context.analytics.LocationAnalytics
import com.flybits.context.analytics.LocationAnalytics.*
import com.flybits.context.models.ContextData
import com.flybits.context.plugins.FlybitsContextPlugin
import com.flybits.context.services.FlybitsContextPluginsWorker
import com.google.android.gms.location.FusedLocationProviderClient
import com.google.android.gms.location.LocationServices
import com.google.android.gms.tasks.Tasks
import java.util.concurrent.ExecutionException

const val TAG = "PluginLocation"

/**
 * The [LocationContextPluginWorker] is a worker responsible for retrieving the last location.
 * This will be triggered by `WorkManager` on start of location plugin to fetch the last location after specified interval.
 * The minimum interval for the worker to execute is 900 seconds.
 * Permissions required are : ACCESS_FINE_LOCATION and ACCESS_COARSE_LOCATION.
 */
class LocationContextPluginWorker(var context: Context, workerParameters: WorkerParameters) :
    FlybitsContextPluginsWorker(context, workerParameters) {

    private var refreshTime: Long = 0
    private lateinit var fusedLocationClient: FusedLocationProviderClient
    private var isGpsEnabled = false
    private var isNetworkEnabled = false
    private lateinit var locationAnalytics: LocationAnalytics
    private lateinit var sharedElements: SharedElements

    override fun initialize(bundle: Bundle?) {
        refreshTime = inputData.getLong(FlybitsContextPlugin.EXTRA_MINIMUM_REFRESH_TIME, 60)
        fusedLocationClient = LocationServices.getFusedLocationProviderClient(applicationContext)
        locationAnalytics = LocationAnalytics(context)
        sharedElements = SharedElementsFactory.get(context)

        var permissionIsNever = false // indicate if the permission is never
        val listOfPermissions = getRequiredPermissions()
        listOfPermissions.forEach { permission ->
            if (ContextCompat.checkSelfPermission(
                    context,
                    permission
                ) != PackageManager.PERMISSION_GRANTED
            ) {
                permissionIsNever = true
            }
        }
        if (permissionIsNever) {
            if (sharedElements.setLocationPermission(Action.NEVER.actionName)) {
                locationAnalytics.trackNever()
            }
        }
    }

    override fun isSupported(): Boolean {
        return true
    }

    override fun getData(): ContextData {
        Logger.appendTag(TAG).d("Fetching Location Data ...")
        try {
            val locationManager = context.getSystemService(Context.LOCATION_SERVICE)
            if (locationManager is LocationManager) {
                isNetworkEnabled =
                    locationManager.isProviderEnabled(LocationManager.NETWORK_PROVIDER)

                isGpsEnabled = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                    locationManager.isLocationEnabled
                } else {
                    locationManager.isProviderEnabled(LocationManager.GPS_PROVIDER)
                            || locationManager.isProviderEnabled(LocationManager.PASSIVE_PROVIDER)
                }
            } else throw NullPointerException("getSystemService() returns null")
            if (isGpsEnabled && isNetworkEnabled) {
                // Using Tasks.await to get the result synchronously without callback.
                val lastLocation = Tasks.await(fusedLocationClient.lastLocation)
                return if (lastLocation != null) {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                        val always = (ContextCompat.checkSelfPermission(
                            context,
                            Manifest.permission.ACCESS_BACKGROUND_LOCATION
                        ) == PackageManager.PERMISSION_GRANTED)
                        if (always) {
                            if (sharedElements.setLocationPermission(Action.ALWAYS.actionName)) {
                                locationAnalytics.trackAlways()
                            }
                        } else {
                            if (sharedElements.setLocationPermission(Action.INUSE.actionName)) {
                                locationAnalytics.trackInUse()
                            }
                        }
                    } else {
                        if (sharedElements.setLocationPermission(Action.ALWAYS.actionName)) {
                            locationAnalytics.trackAlways()
                        }
                    }
                    LocationData(lastLocation)
                } else {
                    throw FlybitsException("Location Data was not retrieved. Please ensure you have access to Google Play Services on the device.")
                }
            } else throw FlybitsException("Location Data was not retrieved. Please ensure both GPS and Network provider are enabled on the device.")
        } catch (e: ExecutionException) {
            throw Exception(e.message)
        } catch (e: InterruptedException) {
            throw Exception(e.message)
        }
    }

    override fun getRequiredPermissions(): Array<String> {
        return arrayOf(
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.ACCESS_COARSE_LOCATION
        )
    }
}