package com.flybits.context.plugins.language;

import android.os.Parcel;
import android.os.Parcelable;

import com.flybits.context.models.ContextData;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Locale;

/**
 * The {@code LanguageData} that indicates which language has been set on the device based on
 * ISO 639-1 standard where language code are indicated by a 2-letter code.
 */
public class LanguageData extends ContextData implements Parcelable {

    private String language;

    /**
     * Constructor used to define the language code based on the default language set on the device.
     */
    public LanguageData(){
        super();
        language    = getDefaultCode();
    }

    /**
     * Constructor used to define the language code based on a preset language code.
     *
     * @param language The 2-letter language code used define the language of the device.
     */
    public LanguageData(String language){
        super();
        this.language = language;
    }

    /**
     * Constructor used for un-flattening a {@code LanguageData} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code LanguageData} parcel.
     */
    protected LanguageData(Parcel in){
        super();
        this.language = in.readString();
        setTime(in.readLong());
    }

    /**
     * Get the language code that indicates the current language of the device.
     *
     * @return The 2-letter language code that indicates which language is set for device.
     */
    public String getLanguage() {
        return language;
    }

    @Override
    public String toJson() {

        JSONObject object=new JSONObject();
        try {
            object.put("language", language);
        }catch (JSONException exception){}

        return object.toString();
    }

    @Override
    public void fromJson(String json) {
        try {
            JSONObject jsonObj  = new JSONObject(json);
            language            = jsonObj.getString("language");

        }catch (JSONException exception){}
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        LanguageData that = (LanguageData) o;
        return language.equals(that.language);
    }

    @Override
    public String getPluginID() {
        return "ctx.sdk.device";
    }

    @Override
    public String toString(){
        return toJson();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(language);
        dest.writeLong(getTime());
    }

    /**
     * Parcelable.Creator that instantiates {@code LanguageData} objects
     */
    public static final Creator<LanguageData> CREATOR = new Creator<LanguageData>() {
        public LanguageData createFromParcel(Parcel in) {
            return new LanguageData(in);
        }

        public LanguageData[] newArray(int size) {
            return new LanguageData[size];
        }
    };

    static String getDefaultCode(){
        return Locale.getDefault().getLanguage();
    }
}
