package com.flybits.context.plugins.activity;

import android.os.Parcel;
import android.os.Parcelable;

import com.flybits.context.models.ContextData;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * The {@code ActivityData} class is responsible for storing information about which activity is
 * currently being performed by the device at any given period of time.
 */
public class ActivityData extends ContextData implements Parcelable {

    /**
     * The {@code ActivityType} indicates which type of activity (Walking, Driving, etc.) the user
     * is currently performing.
     */
    public enum ActivityType {

        /**
         * Indicates that an unknown activity/action is currently being performed.
         */
        UNKNOWN(-99),

        /**
         * Indicates that the user is currently still or not moving.
         */
        STILL(1),

        /**
         * Indicates that the user is currently walking.
         */
        WALKING(2),

        /**
         * Indicates that the user is currently running.
         */
        RUNNING(3),

        /**
         * Indicates that the user is currently traveling in a vehicle.
         */
        IN_VEHICLE(4),

        /**
         * Indicates that the user is currently traveling by bicycle.
         */
        ON_BICYCLE(5);

        private final int key;

        /**
         * Constructor that defines the key for each {@code ActivityType} option.
         *
         * @param key the integer value representing each {@code ActivityType} option.
         */
        ActivityType(int key) {
            this.key = key;
        }

        /**
         * Get the integer representation for the {@code ActivityType} option.
         *
         * @return integer representation of the {@code ActivityType} option.
         */
        public int getKey() {
            return this.key;
        }

        /**
         * Get the {@code ActivityType} enum value corresponding to an integer representation.
         *
         * @param key the integer representation of the {@code ActivityType} enum.
         *
         * @return The {@code ActivityType} enum for the integer representation.
         */
        public static ActivityData.ActivityType fromKey(int key) {
            for(ActivityData.ActivityType type : ActivityData.ActivityType.values()) {
                if(type.getKey() == key) {
                    return type;
                }
            }
            return UNKNOWN;
        }
    }

    private boolean cycling;
    private boolean driving;
    private boolean running;
    private boolean stationary;
    private boolean unknown;
    private boolean walking;

    /**
     * Default constructor needed for generics instantiation
     */
    public ActivityData(){}

    /**
     * Constructor used for un-flattening a {@code ActivityData} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code ActivityData} parcel.
     */
    protected ActivityData(Parcel in){
        super();
        cycling      = in.readInt() == 1;
        walking      = in.readInt() == 1;
        driving      = in.readInt() == 1;
        running      = in.readInt() == 1;
        unknown      = in.readInt() == 1;
        stationary   = in.readInt() == 1;
        setTime(in.readLong());
    }

    /**
     * Constructor that initializes all the different states of the {@code ActivityData} object.
     *
     * @param cycling Indicates true if the user is currently cycling.
     * @param driving Indicates true if the user is currently driving.
     * @param running Indicates true if the user is currently running.
     * @param stationary Indicates true if the user is currently stationary.
     * @param unknown Indicates true if current state of the user is unknown.
     * @param walking Indicates true if the user is currently walking.
     */
    public ActivityData(boolean cycling, boolean driving, boolean running, boolean stationary,
                        boolean unknown, boolean walking){
        super();
        this.cycling    = cycling;
        this.driving    = driving;
        this.running    = running;
        this.stationary = stationary;
        this.unknown    = unknown;
        this.walking    = walking;
    }

    /**
     * Indicates whether or not the user is currently {@link ActivityType#ON_BICYCLE}.
     *
     * @return true if the user is cycling, false otherwise.
     */
    public boolean isCycling() {
        return cycling;
    }

    /**
     * Indicates whether or not the user is currently {@link ActivityType#IN_VEHICLE}.
     *
     * @return true if the user is driving, false otherwise.
     */
    public boolean isDriving() {
        return driving;
    }

    /**
     * Indicates whether or not the user is currently {@link ActivityType#RUNNING}.
     *
     * @return true if the user is running, false otherwise.
     */
    public boolean isRunning() {
        return running;
    }

    /**
     * Indicates whether or not the user is currently {@link ActivityType#STILL}.
     *
     * @return true if the user is standing still, false otherwise.
     */
    public boolean isStationary() {
        return stationary;
    }

    /**
     * Indicates whether or not the user is currently {@link ActivityType#UNKNOWN}.
     *
     * @return true if the user is unknown, false otherwise.
     */
    public boolean isUnknown() {
        return unknown;
    }

    /**
     * Indicates whether or not the user is currently {@link ActivityType#WALKING}.
     *
     * @return true if the user is walking, false otherwise.
     */
    public boolean isWalking() {
        return walking;
    }

    @Override
    public String toJson() {

        JSONObject object=new JSONObject();
        try {
            object.put("cycling", cycling);
            object.put("walking", walking);
            object.put("driving", driving);
            object.put("running", running);
            object.put("unknown", unknown);
            object.put("stationary", stationary);
        }catch (JSONException exception){}

        return object.toString();
    }

    @Override
    public void fromJson(String json) {
        try {
            JSONObject jsonObj = new JSONObject(json);
            cycling            = jsonObj.getBoolean("cycling");
            walking            = jsonObj.getBoolean("walking");
            driving            = jsonObj.getBoolean("driving");
            running            = jsonObj.getBoolean("running");
            unknown            = jsonObj.getBoolean("unknown");
            stationary         = jsonObj.getBoolean("stationary");

        }catch (JSONException exception){}
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ActivityData that = (ActivityData) o;

        if (cycling != that.cycling) return false;
        if (driving != that.driving) return false;
        if (running != that.running) return false;
        if (stationary != that.stationary) return false;
        if (unknown != that.unknown) return false;
        return walking == that.walking;
    }

    @Override
    public String getPluginID() {
        return "ctx.sdk.activity";
    }

    @Override
    public String toString(){
        return toJson();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeInt(cycling ? 1 : 0);
        dest.writeInt(walking ? 1 : 0);
        dest.writeInt(driving ? 1 : 0);
        dest.writeInt(running ? 1 : 0);
        dest.writeInt(unknown ? 1 : 0);
        dest.writeInt(stationary ? 1 : 0);
        dest.writeLong(getTime());
    }

    /**
     * Parcelable.Creator that instantiates {@code ActivityData} objects
     */
    public static final Creator<ActivityData> CREATOR = new Creator<ActivityData>() {
        public ActivityData createFromParcel(Parcel in) {
            return new ActivityData(in);
        }

        public ActivityData[] newArray(int size) {
            return new ActivityData[size];
        }
    };
}
